% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coersion-tk_zoo.R
\name{tk_zoo}
\alias{tk_zoo}
\alias{tk_zoo_}
\title{Coerce time series objects and tibbles with date/date-time columns to xts.}
\usage{
tk_zoo(data, select = NULL, date_var = NULL, silent = FALSE, ...)

tk_zoo_(data, select = NULL, date_var = NULL, silent = FALSE, ...)
}
\arguments{
\item{data}{A time-based tibble or time-series object.}

\item{select}{\strong{Applicable to tibbles and data frames only}.
The column or set of columns to be coerced to \code{ts} class.}

\item{date_var}{\strong{Applicable to tibbles and data frames only}.
Column name to be used to \code{order.by}.
\code{NULL} by default. If \code{NULL}, function will find the date or date-time column.}

\item{silent}{Used to toggle printing of messages and warnings.}

\item{...}{Additional parameters to be passed to \code{xts::xts()}. Refer to \code{xts::xts()}.}
}
\value{
Returns a \code{zoo} object.
}
\description{
Coerce time series objects and tibbles with date/date-time columns to xts.
}
\details{
\code{tk_zoo} is a wrapper for \code{zoo::zoo()} that is designed
to coerce \code{tibble} objects that have a "time-base" (meaning the values vary with time)
to \code{zoo} class objects. There are three main advantages:
\enumerate{
\item Non-numeric columns that are not removed via \code{select} are dropped and the user is warned.
This prevents an error or coercion issue from occurring.
\item The date column is auto-detected if not specified by \code{date_var}. This takes
the effort off the user to assign a date vector during coercion.
\item \code{ts} objects are automatically coerced if a "timetk index" is present. Refer to \code{\link[=tk_ts]{tk_ts()}}.
}

The \code{select} argument can be used to select subsets
of columns from the incoming data.frame.
Only columns containing numeric data are coerced.
The \code{date_var} can be used to specify the column with the date index.
If \code{date_var = NULL}, the date / date-time column is interpreted.
Optionally, the \code{order.by} argument from the underlying \code{zoo::zoo()} function can be used.
The user must pass a vector of dates or date-times if \code{order.by} is used.
\emph{Important Note: The \code{...} arguments are passed to \code{xts::xts()}, which
enables additional information (e.g. time zone) to be an attribute of the \code{zoo} object.}

For non-data.frame object classes (e.g. \code{xts}, \code{zoo}, \code{timeSeries}, etc) the objects are coerced
using \code{zoo::zoo()}.

\code{tk_zoo_} is a nonstandard evaluation method.
}
\examples{
library(tidyverse)
library(timetk)

### tibble to zoo: Comparison between tk_zoo() and zoo::zoo()
data_tbl <- tibble::tibble(
    date = seq.Date(as.Date("2016-01-01"), by = 1, length.out = 5),
    x    = rep("chr values", 5),
    y    = cumsum(1:5),
    z    = cumsum(11:15) * rnorm(1))

# zoo: Characters will cause error; order.by must be passed a vector of dates
zoo::zoo(data_tbl[,-c(1,2)], order.by = data_tbl$date)

# tk_zoo: Character columns dropped with a warning; No need to specify dates (auto detected)
tk_zoo(data_tbl)

# ts can be coerced back to zoo
data_tbl \%>\%
    tk_ts(start = 2016, freq = 365) \%>\%
    tk_zoo()


### Using select and date_var
tk_zoo(data_tbl, select = y, date_var = date)


### NSE: Enables programming
date_var <- "date"
select   <- "y"
tk_zoo_(data_tbl, select = select, date_var = date_var)

}
\seealso{
\code{\link[=tk_tbl]{tk_tbl()}}, \code{\link[=tk_xts]{tk_xts()}}, \code{\link[=tk_zooreg]{tk_zooreg()}}, \code{\link[=tk_ts]{tk_ts()}}
}
