% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unbalanced.R, R/unbalanced_wpp.R
\name{unbalanced}
\alias{unbalanced}
\alias{unbalanced.pgrid}
\alias{unbalanced.wpp}
\title{Unbalanced Optimal Transport Between Two Objects}
\usage{
unbalanced(a, b, ...)

\method{unbalanced}{pgrid}(
  a,
  b,
  p = 1,
  C = NULL,
  method = c("networkflow", "revsimplex"),
  output = c("dist", "all", "rawres"),
  threads = 1,
  ...
)

\method{unbalanced}{wpp}(
  a,
  b,
  p = 1,
  C = NULL,
  method = c("networkflow", "revsimplex"),
  output = c("dist", "all", "rawres"),
  threads = 1,
  ...
)
}
\arguments{
\item{a, b}{objects of class \code{\link{pgrid}} or \code{\link{wpp}} that are compatible.}

\item{...}{other arguments.}

\item{p}{a power \eqn{\geq 1} applied to the transport and disposal costs. The order
of the resulting unbalanced Wasserstein metric.}

\item{C}{The base disposal cost (without the power \code{p})}

\item{method}{one of \code{"networkflow"} and \code{"revsimplex"}, specifing the algorithm used. See details.}

\item{output}{character. One of "dist", "all" and "rawres". Determines what the function
returns: only the unbalanced Wasserstein distance; all available information about the 
transport plan and the extra mass; or the raw result obtained by the networkflow algorithm.
The latter is the same format as in the \code{transport} function with option \code{fullreturn=TRUE}.
The choice \code{output = "rawres"} is mainly intended for internal use.}

\item{threads}{an integer specifying the number of threads for parallel computing in connection with
the networkflow method.}
}
\value{
If \code{output = "dist"} a single numeric, the unbalanced \eqn{(p,C)}-Wasserstein distance.
Otherwise a list. If \code{output = "all"} the list is of class \code{ut_pgrid} or \code{ut_wpp} according
to the class of the objects \code{a} and \code{b}. It has \code{a}, \code{b}, \code{p}, \code{C} as attributes and 
the following components:
\item{dist}{same as for \code{output = "dist"}.}
\item{plan}{an optimal transport plan. This is a data frame with columns \code{from}, \code{to} and \code{mass}
 that specifies from which element of \code{a} to which element of \code{b} what amount of mass is sent.
 \code{from} and \code{to} are specified as vector indices in terms of the usual column major enumeration
 of the matrices a$mass and b$mass. The plan can be plotted via \code{plot.pgrid(a, b, plan)}.}
\item{atrans, btrans}{matrices (pgrid) or vectors (wpp) specifying the masses transported from each point and to each point,
 respectively. Corresponds to \eqn{(\pi^{(1)}_x)_{x \in S}} and \eqn{(\pi^{(2)}_y)_{y \in S}} above.}
\item{aextra, bextra}{matrices (pgrid) or vectors (wpp) specifying the amount of mass at each point of \code{a} and \code{b},
respectively, that cannot be transported and needs to be disposed of. Corresponds to
\eqn{(a_x - \pi^{(1)}_x)_{x \in S}} and \eqn{(b_y - \pi^{(2)}_y)_{y \in S}}.}
\item{inplace}{(pgrid only) a matrix specifying the amount of mass at each point that can stay in place. Corresponds
to \eqn{(\pi_{x,x})_{x \in S}}.}
 
Note that \code{atrans + aextra + inplace} (pgrid) or \code{atrans + aextra} (wpp)must be equal
to \code{a$mass} and likewise for b.
A warning occurs if this is not the case (which may indeed happen from time to time for method
revsimplex, but the error reported should be very small).
}
\description{
Compute optimal transport between unnormalized images / mass distributions on grids 
(\code{pgrid} objects) or between mass distributions on general point patterns
(\code{wpp} objects) under the option that mass can be dispose of. Transport cost
per unit is the Euclidean distance of the transport to the \code{p}-th power.
Disposal cost per unit is \code{C^p}.\cr
}
\details{
Given two non-negative mass distributions \eqn{a=(a_x)_{x \in S}}, \eqn{b=(a_y)_{y \in S}}
on a set \eqn{S} (a pixel grid / image if \code{a}, \code{b} are of class \code{pgrid} or a more
general weighted point pattern if \code{a}, \code{b} are of class \code{wpp}), this function minimizes the functional
\deqn{\sum_{x,y \in S} \pi_{x,y} d(x,y)^p + C^p \bigl( \sum_{x \in S} (a_x - \pi^{(1)}_x) + \sum_{y \in S} (b_y - \pi^{(2)}_y) \bigr)}
over all \eqn{(\pi_{x,y})_{x,y \in S}} satisfying
\deqn{0 \leq \pi^{(1)}_x := \sum_{y \in S} \pi_{x,y} \leq a_x \ \textrm{and} \ 0 \leq \pi^{(2)}_y := \sum_{x \in S} \pi_{x,y} \leq b_y.}

Thus \eqn{\pi_{x,y}} denotes the amount of mass transported from \eqn{x} to \eqn{y}, whereas \eqn{\pi^{(1)}_x}
and \eqn{\pi^{(2)}_y} are the total mass transported away from \eqn{x} and total mass transported to \eqn{y}, respectively.
Accordingly \eqn{\sum_{x \in S} (a_x - \pi^{(1)}_x)} and \eqn{\sum_{y \in S} (b_y - \pi^{(2)}_y)} are the total
amounts of mass of \eqn{a} and \eqn{b}, respectively, that need to be disposed of.

The minimal value of the functional above taken to the \eqn{1/p} is what we refer to as unbalanced
\eqn{(p,C)}-Wasserstein metric. This metric is used, in various variants, in an number of research papers.
See Heinemann et al. (2022) and the references therein and Müller et al. (2022), Remark 3. We follow the
convention of the latter paper regarding the parametrization and the use of the term \emph{unbalanced Wasserstein metric}.

The practical difference between the two methods "networkflow" and "revsimplex" can 
roughly described as follows. The former is typically faster for large examples (for \code{pgrid} objects 64x64
and beyond), especially if several threads are used. The latter is typically faster
for smaller examples (which may be relevant if pairwise transports between many objects
are computed) and it guarantees a sparse(r) solution, i.e. at most \eqn{m+n+1} individual
transports, where \eqn{m} and \eqn{n} are the numbers of non-zero masses in \code{a} and \code{b}, respectively).
Note however that due to the implementation the revsimplex algorithm is a little less
precise (roughly within 1e-7 tolerance). For more details on the algorithms see \code{\link{transport}}.
}
\examples{
a <- pgrid(matrix(1:12, 3, 4))
b <- pgrid(matrix(c(9:4, 12:7), 3, 4))
res1 <- unbalanced(a, b, 1, 0.5, output="all")
res2 <- unbalanced(a, b, 1, 0.3, output="all")
plot(a, b, res1$plan, angle=20, rot=TRUE)
plot(a, b, res2$plan, angle=20, rot=TRUE)
par(mfrow=c(1,2))
matimage(res2$aextra, x = a$generator[[1]], y = a$generator[[2]])
matimage(res2$bextra, x = b$generator[[1]], y = b$generator[[2]])

set.seed(31)
a <- wpp(matrix(runif(8),4,2), 3:6)
b <- wpp(matrix(runif(10),5,2), 1:5)
res1 <- unbalanced(a, b, 1, 0.5, output="all")
res2 <- unbalanced(a, b, 1, 0.3, output="all")
plot(a, b, res1$plan)
plot(a, b, res2$plan)

}
\references{
Florian Heinemann, Marcel Klatt and Axel Munk (2022).\cr
            Kantorovich-Rubinstein distance and barycenter for finitely supported measures: Foundations and Algorithms.\cr
            Arxiv preprint.\cr
            \doi{10.48550/arXiv.2112.03581}\cr
            \cr
            Raoul Müller, Dominic Schuhmacher and Jorge Mateu (2020).\cr
            Metrics and barycenters for point pattern data
            Statistics and Computing 30, 953-972.\cr
            \doi{10.1007/s11222-020-09932-y}
}
\seealso{
\code{\link{plot.ut_pgrid}} and \code{\link{plot.ut_wpp}}, which can plot the various components of the list obtained for \code{output="all"}.
}
