% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ppdist.R
\name{ppdistnet}
\alias{ppdistnet}
\alias{ttdistnet}
\title{Compute Distance Between Two Point Patterns on a Network}
\usage{
ppdistnet(
  dmat,
  xi = NULL,
  eta = NULL,
  penalty = 1,
  type = c("tt", "rtt", "TT", "RTT"),
  ret_matching = FALSE,
  p = 1,
  precision = NULL
)
}
\arguments{
\item{dmat}{a matrix specifying in its \eqn{(i,j)}-th entry the shortest-path distance from the
i-th point of \eqn{\xi}{xi} to the j-th point of  \eqn{\eta}{eta}
OR the distance matrix of a whole network. In the latter case arguments \eqn{\xi}{xi} and
\eqn{\eta}{eta} have to be specified.}

\item{xi}{a vector specifying the vertex-indices of \eqn{\xi}{xi}, only needed if \code{dmat} is the 
distance matrix of a whole network.}

\item{eta}{a vector specifying the vertex-indices of \eqn{\eta}{eta}, only needed if \code{dmat} is the 
distance matrix of a whole network.}

\item{penalty}{a positive number. The penalty for adding/deleting points.}

\item{type}{either \code{"tt"}/\code{"TT"} for the transport-transform metric
or \code{"rtt"}/\code{"RTT"} for the relative transport-transform metric.}

\item{ret_matching}{Logical. Shall the optimal point matching be returned?}

\item{p}{a number \eqn{>0}. The matching is chosen such that the \code{p}-th
order sum (\eqn{\ell_p}{l_p}-norm) is minimized.}

\item{precision}{a small positive integer value. The precision of the computations, which
are currently performed in integers. After correcting for the penalty, \code{dmat^p}
is divided by its largest entry, multiplied by \code{10^precision} and rounded to
compute the optimal matching. The default value \code{NULL} chooses maximal
integer precision possible, which is \code{precision = 9} on almost all systems.}
}
\value{
The corresponding distance between the point patterns if \code{ret_matching}
        is \code{FALSE}. 
        
        Otherwise a list with components \code{dist} containing
        this distance and two vectors \code{target1, target2} of integers, where
        \code{target}\eqn{i} specifies the indices of the points in the other pattern
        that the points of the \eqn{i}-th pattern are matched to and \code{NA} every
        time a point is deleted.

        There may be a minus in front of an index, where
        \code{-j} indicates that the corresponding pairing with point \code{j}
        would be over a distance of more than \eqn{2^{1/p} \cdot 
        \code{penalty}}{2^{1/p} * penalty}. This is 
        equivalent to saying that the corresponding point of the first pattern
        is deleted and the \eqn{j}-th point of the second pattern is added. 
        
        Note that having more than one minus implies that the matching is non-unique.
}
\description{
Based on an arbitrary matrix of "distances" on a network, this function computes versions
of the transport-transform distance between two point patterns \eqn{\xi}{xi} and \eqn{\eta}{eta}
on this network.
}
\details{
This function provides a more convenient way for computing (relative)
         transport-transform distances on networks if the points of the patterns are given in
         terms of indices of network vertices. If \code{dmat} contains only the distances
         between the points of \eqn{\xi}{xi} and \eqn{\eta}{eta}, this function
         does the same as \code{\link{ppdist}}.
}
\examples{
  set.seed(123456)
  nvert <- 100 #number of vertices in the network
  lambda <- 0.5 #expected number of points per unit length
  ln <- delaunayNetwork(runifpoint(nvert)) #create an artificial network
  ppnetwork <- rpoislpp(lambda, ln, nsim = 2)
    #simulate two point patterns on the network

  plot(ppnetwork[[1]]$domain, cex=0.5, main="")
  plot(as.ppp(ppnetwork[[1]]),vpch=1,col=2,add=TRUE)
  plot(as.ppp(ppnetwork[[2]]),vpch=1,col=4,add=TRUE)

  res <- netsplit(ln, ppnetwork)
    #incorporate data point patterns into the network
    #calculating all pairwise distances between vertices
    #and creating matrix of vertex-indices of data point patterns
  
  xi <- res$ppmatrix[1:npoints(ppnetwork[[1]]), 1]
  eta <- res$ppmatrix[1:npoints(ppnetwork[[2]]), 2]
  res2 <- ppdistnet(res$network$dpath, xi = xi, eta = eta,
                    penalty = 1, type = "tt", ret_matching = TRUE, p = 1)
  res2

}
\seealso{
\code{\link{ppdist}}
}
\author{
Raoul Müller  \email{raoul.mueller@uni-goettingen.de}\cr
        Dominic Schuhmacher \email{schuhmacher@math.uni-goettingen.de}
}
