% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/umx_build_umxSexLim.R
\name{umxSexLim}
\alias{umxSexLim}
\title{Multivariate sex limitation twin model}
\usage{
umxSexLim(
  name = "sexlim",
  selDVs,
  mzmData,
  dzmData,
  mzfData,
  dzfData,
  dzoData,
  sep = NA,
  A_or_C = c("A", "C"),
  sexlim = c("Nonscalar", "Scalar", "Homogeneity"),
  dzAr = 0.5,
  dzCr = 1,
  autoRun = getOption("umx_auto_run"),
  tryHard = c("no", "yes", "ordinal", "search"),
  optimizer = NULL
)
}
\arguments{
\item{name}{The name of the model (Default = "sexlim")}

\item{selDVs}{BASE NAMES of the variables in the analysis. You MUST provide sep.}

\item{mzmData}{Dataframe containing the MZ male data.}

\item{dzmData}{Dataframe containing the DZ male data.}

\item{mzfData}{Dataframe containing the MZ female data.}

\item{dzfData}{Dataframe containing the DZ female data.}

\item{dzoData}{Dataframe containing the DZ opposite-sex data (be sure and get in right order).}

\item{sep}{Suffix used for twin variable naming. Allows using just the base names in selVars.}

\item{A_or_C}{Whether to model sex-limitation on A or on C. (Defaults to "A").}

\item{sexlim}{Which model type: "Nonscalar" (default), "Scalar", or "Homogeneity".}

\item{dzAr}{The DZ genetic correlation (defaults to .5, vary to examine assortative mating).}

\item{dzCr}{The DZ "C" correlation (defaults to 1: set to .25 to make an ADE model).}

\item{autoRun}{Whether to mxRun the model (default TRUE: the estimated model will be returned).}

\item{tryHard}{Default ('no') uses normal mxRun. "yes" uses mxTryHard. Other options: "ordinal", "search"}

\item{optimizer}{optionally set the optimizer. Default (NULL) does nothing.}
}
\value{
\itemize{
\item \code{\link[=mxModel]{mxModel()}} of subclass mxModel.CFSexLim
}
}
\description{
Multivariate twin analysis allowing for sex limitation (factors operate differently in males
vs. females) based on a correlated factors model. With 5-groups of twins, this model allows
for both Quantitative and Qualitative Sex-Limitation.

\emph{Quantitative} differences refer to different amounts of phenotypic variance produced by
the same A, C, or E components when operating in one sex compared to the other sex.

\emph{Qualitative} differences refer to phenotypic variance attributable to an A, C, or E
component which operates in one sex one but not in the other.

The correlation approach ensures that variable order does not affect the ability
of the model to account for DZOS data.

\strong{1. Nonscalar Sex Limitation}

Allow quantitative (distinct male and female paths) and qualitative sex differences
on A or C. Allows distinct between variable correlations (\code{Ra}, \code{Rc} and \code{Re})
for males and for females. Male-Female correlations also free (\code{Rao} or \code{Rco} free in DZO group).

\strong{2. Scalar Sex Limitation}

Quantitative sex differences only (distinct Male and female paths).
Just one set of Ra, Rc and Re between variables (same for males and females)

\strong{3. Homogeneity}

This is the model assumed by the basic ACE model: equal variance components in both sexes.
Different means may be allowed for males and females.
}
\details{
\strong{A or C}

Due to limitations on the degrees of freedom allowed by the twin model, we can model
qualitative sex differences for only one of A or C at a time.

\emph{notes}: There is a half-way house model of heterogeneity in which a, c, and e components are scaled by a
scalar constant in one sex.

\emph{General restrictions}: Assumes means and variances can be equated across birth order within zygosity groups.
}
\examples{
# =========================
# = Load and Process Data =
# =========================
\dontrun{
require(umx)
data("us_skinfold_data")
# Rescale vars
us_skinfold_data[, c('bic_T1', 'bic_T2')] = us_skinfold_data[, c('bic_T1', 'bic_T2')]/3.4
us_skinfold_data[, c('tri_T1', 'tri_T2')] = us_skinfold_data[, c('tri_T1', 'tri_T2')]/3
us_skinfold_data[, c('caf_T1', 'caf_T2')] = us_skinfold_data[, c('caf_T1', 'caf_T2')]/3
us_skinfold_data[, c('ssc_T1', 'ssc_T2')] = us_skinfold_data[, c('ssc_T1', 'ssc_T2')]/5
us_skinfold_data[, c('sil_T1', 'sil_T2')] = us_skinfold_data[, c('sil_T1', 'sil_T2')]/5

# Data for each of the 5 twin-type groups
mzmData = subset(us_skinfold_data, zyg == 1)
mzfData = subset(us_skinfold_data, zyg == 2)
dzmData = subset(us_skinfold_data, zyg == 3)
dzfData = subset(us_skinfold_data, zyg == 4)
dzoData = subset(us_skinfold_data, zyg == 5)

umxSummarizeTwinData(us_skinfold_data, selVars="bic",zyg="zyg", sep="_T",
		MZFF=2, DZFF=4, MZMM=1, DZMM=3, DZOS=5
)

# ==========================
# = Run univariate example =
# ==========================

m1 = umxSexLim(selDVs = "bic", sep = "_T", A_or_C = "A", tryHard = "yes",
	mzmData = mzmData, dzmData = dzmData, 
	mzfData = mzfData, dzfData = dzfData, 
	dzoData = dzoData
)

# Drop qualitative sex limitation
m1a = umxModify(m1, regex = "^Rao_", value=1, name = "no_qual", comparison = TRUE)


# Equate a, ac, and try ace across m & f in scalar model
m1b = umxModify(m1a, regex = "^a[fm]_", newlabels="a_", name = "eq_a_no_qual", comparison = TRUE)
m1c = umxModify(m1b, regex = "^c[fm]_", newlabels="c_", name = "eq_ac_no_qual", comparison = TRUE)
m1d = umxModify(m1c, regex = "^e[fm]_", newlabels="e_", name = "eq_ace_no_qual", comparison = TRUE)
umxCompare(m1, c(m1a, m1b, m1c, m1d))

# ============================
# = Scalar Sex Limitation =
# ============================

m2 = umxSexLim(selDVs = "bic", sep = "_T", sexlim = "Scalar", tryHard = "yes",
	mzmData = mzmData, dzmData = dzmData, 
	mzfData = mzfData, dzfData = dzfData, 
	dzoData = dzoData
) 

# Show our manual drop of qualitative is the same as umxSexLim with sexlim= "scalar"s
umxCompare(m1a, m2)

# ===============
# = Homogeneity =
# ===============

m3 = umxSexLim(selDVs = "bic", sep = "_T", sexlim = "Homogeneity", tryHard = "yes",
	mzmData = mzmData, dzmData = dzmData, 
	mzfData = mzfData, dzfData = dzfData, 
	dzoData = dzoData
)
umxCompare(m1, c(m2, m3))

# ===========================================
# = Bivariate example with manual reduction =
# ===========================================
m1 = umxSexLim(selDVs = c("bic", "tri"), sep = "_T", A_or_C = "A", tryHard="yes",
	mzmData = mzmData, dzmData = dzmData, 
	mzfData = mzfData, dzfData = dzfData, 
	dzoData = dzoData
)

# Scalar sex limitation (same correlation among components for m and f)
m2 = umxSexLim(selDVs = c("bic", "tri"), sep = "_T", 
	A_or_C = "A", tryHard="yes", sexlim="Scalar",
	mzmData = mzmData, dzmData = dzmData, 
	mzfData = mzfData, dzfData = dzfData, 
	dzoData = dzoData
)
# Drop qualitative sex limitation
#  Distinct af and am (& c & e), but shared Ra (& Rc & Re) between variables
#  	i.e., same correlations for males and females.
m1a = umxModify(m1 , regex = "^Ra[mfo]_", newlabels="^Ra_", name = "no_qual_a", comparison = TRUE)
m1b = umxModify(m1a, regex = "^Rc[mfo]_", newlabels="^Rc_", name = "no_qual_ac", comparison = TRUE)
m1c = umxModify(m1b, regex = "^Re[mfo]_", newlabels="^Re_", name = "no_qual_ace", comparison = TRUE)
umxCompare(m1, c(m1a, m1b, m1c, m2))

# In one smart regular expression
m2 = umxModify(m1, regex = "^R([ace])[fmo]_", newlabels = "R\\\\1_", 
  name = "scalar", comparison = TRUE)

# Equate a, ac, and try ace across m & f in scalar model
m2a = umxModify(m2 , regex = "^a[fm]_", newlabels="a_", name = "eq_a_no_qual"  , comparison = TRUE)
m2b = umxModify(m2a, regex = "^c[fm]_", newlabels="c_", name = "eq_ac_no_qual" , comparison = TRUE)
m2c = umxModify(m2b, regex = "^e[fm]_", newlabels="e_", name = "eq_ace_no_qual", comparison = TRUE)
umxCompare(m1, c(m1a, m1b, m1c, m1d))

# =============================
# = Run multi-variate example =
# =============================
# Variables for Analysis
selDVs = c('ssc','sil','caf','tri','bic')
selDVs = c('ssc','tri','bic')
m1 = umxSexLim(selDVs = selDVs, sep = "_T", A_or_C = "A", tryHard = "yes",
mzmData = mzmData, dzmData = dzmData, 
 mzfData = mzfData, dzfData = dzfData, dzoData = dzoData
)

m2 = umxSexLim(selDVs = selDVs, sep = "_T", A_or_C = "A", sexlim = "Nonscalar",
	tryHard = "yes",
mzmData = mzmData, dzmData = dzmData, 
 mzfData = mzfData, dzfData = dzfData, dzoData = dzoData
)

# umxSummary(m1)
# summary(m1)
# summary(m1)$Mi
}
}
\references{
\itemize{
\item Neale et al. (2006).
Multivariate genetic analysis of sex-lim and GxE interaction. \emph{Twin Research & Human Genetics}, \strong{9}, pp. 481--489.
}
}
\seealso{
\code{\link[=umxSummarySexLim]{umxSummarySexLim()}}, \code{\link[=umxPlotSexLim]{umxPlotSexLim()}}

Other Twin Modeling Functions: 
\code{\link{power.ACE.test}()},
\code{\link{umxACEcov}()},
\code{\link{umxACEv}()},
\code{\link{umxACE}()},
\code{\link{umxCP}()},
\code{\link{umxDiffMZ}()},
\code{\link{umxDiscTwin}()},
\code{\link{umxDoCp}()},
\code{\link{umxDoC}()},
\code{\link{umxGxE_window}()},
\code{\link{umxGxEbiv}()},
\code{\link{umxGxE}()},
\code{\link{umxIP}()},
\code{\link{umxReduceACE}()},
\code{\link{umxReduceGxE}()},
\code{\link{umxReduce}()},
\code{\link{umxRotate.MxModelCP}()},
\code{\link{umxSimplex}()},
\code{\link{umxSummarizeTwinData}()},
\code{\link{umxSummaryACEv}()},
\code{\link{umxSummaryACE}()},
\code{\link{umxSummaryDoC}()},
\code{\link{umxSummaryGxEbiv}()},
\code{\link{umxSummarySexLim}()},
\code{\link{umxSummarySimplex}()},
\code{\link{umxTwinMaker}()},
\code{\link{umx}}
}
\concept{Twin Modeling Functions}
