\name{Voigt}
\alias{dvoigt}
\alias{rvoigt}
\title{
The Voigt Distribution
}
\description{
Density function and random generation for the Voigt distribution with median equal to \code{mu} and scale parameters equal to \code{sigma} and \code{gamma}. 
}
\usage{
dvoigt(y, mu = 0, sigma = 1, gamma = 1, log = FALSE, complex = FALSE)
rvoigt(n = 1, mu = 0, sigma = 1, gamma = 1)
}
\arguments{
  \item{n}{
The size of the random sample.
}
  \item{y}{
Vector of quantiles.
}
  \item{mu}{
The median of the Voigt.
}
  \item{sigma}{
Voigt first scale parameter.
}
  \item{gamma}{
Voigt second scale parameter.
}
  \item{log}{
logical; if TRUE, densities d are given as log(d).
}
  \item{complex}{
logical; if FALSE, the Voigt density is returned, the real part of the Faddeva function (see Details). Otherwise it returns both the real and the complex part.
}
}
\details{
The Voigt distribution is the convolution of a Normal and a Cauchy. The density function is 
\deqn{
f(y)=\frac{\Re(w(z))}{\sigma \sqrt{2\pi}}, \quad \sigma,\gamma>0 
}
where \eqn{w(z)=e^{-z^2}erfc(-iz)} is the \emph{Faddeeva} function, \eqn{z=\frac{y-\mu +i\gamma}{\sigma\sqrt{2}}}, \code{mu} is the median, \code{sigma} is the standard deviation of the normal component and \code{gamma} is the scale parameter of the Cauchy component. 
If \code{mu}, \code{sigma} and \code{gamma} are not specified they assume the default values of 0, 1 and 1, respectively.
}
\value{
\code{dvoigt} gives the density and \code{rvoigt} generates random deviates.
}
\references{
Kendall, D. G. (1938). The effect of radiation damping and Doppler broadening on the atomic absorption coefficient.
\emph{Zeitschrift fur Astrophysik, Vol. 16, p.308}
\url{https://adsabs.harvard.edu/full/1938ZA.....16..308K}

  Abramowitz, M., and I. A. Stegun (1972). Handbook of Mathematical Functions
  (with Formulas, Graphs, and Mathematical Tables). Dover, New York.  
  \url{https://personal.math.ubc.ca/~cbm/aands/notes.htm}
}
\author{
\packageAuthor{voigt}

}
\note{
\code{rvoigt} is a wrapper of both \code{rnorm} and \code{rcauchy} functions.
\code{dvoigt} uses \code{erfz}, the complex error function, see \code{\link[pracma]{erfz}}.

}

\seealso{
\code{\link{evoigt}}
}
\examples{
dvoigt(0)
# 0.2087093
# plot voigt and gaussian density
x = seq(from=-4,to=4, by=0.01)
plot(x, dvoigt(x), type="l",ylab="",ylim=c(0,0.8))
lines(x,dvoigt(x,0,1/3,1/3),col="blue")
lines(x,dnorm(x),lty=2, col="red")
mtext("dvoigt(x,0,1,1)", adj = 0,col="black")
mtext("dnorm(x)", adj = 1/2,col="red")
mtext("dvoigt(x,0,1/3,1/3)", adj = 1,col="blue")

# compare true and empirical density
rvoigt(1)
x = rvoigt(n=500)
q=quantile(x,0.99)
hist(x[x>-q & x<q], prob=TRUE, breaks=30,ylim=c(0,1.1*dvoigt(0)),main="", xlab="x")
x.grid = seq(-q,q,by=.1)
lines(x.grid, dvoigt(x.grid), type="l", ylab="", xlab="",col="red")
# compare with cauchy and normal density
par(mfrow=c(1,2))
x = rvoigt(n=500, mu = 0,sigma = 1,gamma = 0.1)
q=quantile(x,0.99)
hist(x[x>-q & x<q], prob=TRUE,breaks=20,col="lightgreen",main = "dvoigt(0, 1, 0.1)",xlab="x")
curve(dnorm(x),lty=1, add=TRUE)

y = rvoigt(n=500, mu = 0,sigma = 0.1,gamma = 1)
q=quantile(y,0.99)
hist(y[y>-q & y<q], prob=TRUE,breaks=25,col="lightblue",main = "dvoigt(0, 0.1, 1)",xlab="x")
curve(dcauchy(x),lty=1, add=TRUE)
dev.off()
}
