% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale_index.R
\name{scale_index}
\alias{scale_index}
\title{Scale index of a signal}
\usage{
scale_index(signal = NULL,
                   scalog = NULL,
                   dt = 1,
                   scales = NULL,
                   powerscales = TRUE,
                   s1 = NULL,
                   wname = c("MORLET", "DOG", "PAUL", "HAAR", "HAAR2"),
                   wparam = NULL,
                   waverad = NULL,
                   border_effects = c("BE", "INNER", "PER", "SYM"),
                   makefigure = TRUE,
                   figureperiod = TRUE,
                   plot_scalog = FALSE,
                   xlab = NULL,
                   ylab = "Scale index",
                   main = "Scale Index")
}
\arguments{
\item{signal}{A vector containing the signal whose scale indices are wanted.}

\item{scalog}{A vector containing the scalogram from which the scale indices are going
to be computed. If \code{scalog} is not \code{NULL}, then \code{signal}, \code{waverad}
and \code{border_effects} are not necessary and they are ignored.}

\item{dt}{Numeric. The time step of the signals.}

\item{scales}{A vector containing the wavelet scales at wich the scalogram
is computed. This can be either a vector with all the scales or, following Torrence
and Compo 1998, a vector of 3 elements with the minimum scale, the maximum scale and
the number of suboctaves per octave (in this case, \code{powerscales} must be TRUE in
order to construct power 2 scales using a base 2 logarithmic scale). If \code{scales}
is NULL, they are automatically constructed.}

\item{powerscales}{Logical. It must be TRUE (default) in these cases:
\itemize{
\item If \code{scales} are power 2 scales, i.e. they use a base 2 logarithmic scale.
\item If we want to construct power 2 scales automatically. In this case, \code{scales}
must be \code{NULL}.
\item If we want to construct power 2 scales from \code{scales}. In this case,
\code{length(scales)} must be 3.
}}

\item{s1}{A vector containing the scales \eqn{s_1}. The scale indices are computed in
the intervals \eqn{[s_0,s_1]}, where \eqn{s_0} is the minimum scale in \code{scales}.
If \code{s1} are not power 2 scales, then \code{scales} should not be power 2 scales
either and hence, \code{powerscales} must be \code{FALSE}.}

\item{wname}{A string, equal to "MORLET", "DOG", "PAUL", "HAAR" or "HAAR2". The
difference between "HAAR" and "HAAR2" is that "HAAR2" is more accurate but slower.}

\item{wparam}{The corresponding nondimensional parameter for the wavelet function
(Morlet, DoG or Paul).}

\item{waverad}{Numeric. The radius of the wavelet used in the computations for the cone
of influence. If it is not specified, it is asumed to be \eqn{\sqrt{2}} for Morlet and DoG,
\eqn{1/\sqrt{2}} for Paul and 0.5 for Haar.}

\item{border_effects}{A string, equal to "BE", "INNER", "PER" or "SYM", which indicates
how to manage the border effects which arise usually when a convolution is performed on
finite-lenght signals.
\itemize{
\item "BE": With border effects, padding time series with zeroes.
\item "INNER": Normalized inner scalogram with security margin adapted for each
different scale.
\item "PER": With border effects, using boundary wavelets (periodization of the
original time series).
\item "SYM": With border effects, using a symmetric catenation of the original time
series.
}}

\item{makefigure}{Logical. If TRUE (default), a figure with the scale indices is
plotted.}

\item{figureperiod}{Logical. If TRUE (default), periods are used in the figure instead
of scales.}

\item{plot_scalog}{Logical. If TRUE, it plots the scalogram from which the scale indices
are computed.}

\item{xlab}{A string giving a custom X axis label. If NULL (default) the X label is
either "s1" or "Period of s1" depending on the value of \code{figureperiod}.}

\item{ylab}{A string giving a custom Y axis label.}

\item{main}{A string giving a custom main title for the figure.}
}
\value{
A list with the following fields:
\itemize{
\item \code{si}: A vector with the scale indices.
\item \code{s0}: The scale \eqn{s_0}.
\item \code{s1}: A vector with the scales \eqn{s_1}.
\item \code{smax}: A vector with the scales \eqn{s_{max}}.
\item \code{smin}: A vector with the scales \eqn{s_{min}}.
\item \code{scalog}: A vector with the scalogram from which the scale indices are
computed.
\item \code{scalog_smax}: A vector with the maximum scalogram values \eqn{S(s_{max})}.
\item \code{scalog_smin}: A vector with the minimum scalogram values \eqn{S(s_{min})}.
\item \code{fourierfactor}: A factor for converting scales into periods.
}
}
\description{
This function computes the scale index of a signal in the scale interval
\eqn{[s_0,s_1]}, for a given set of scale parameters \eqn{s_1} and taking \eqn{s_0} as
the minimum scale (see Benítez et al. 2010).

The scale index of a signal in the scale interval \eqn{[s_0,s_1]} is given by the
quotient \deqn{\frac{S(s_{min})}{S(s_{max})},}{S(s_{min})/S(s_{max})} where \eqn{S} is
the scalogram, \eqn{s_{max} \in [s_0,s_1]} is the smallest scale such that
\eqn{S(s)\le S(s_{max})} for all \eqn{s \in [s_0,s_1]}, and
\eqn{s_{min} \in [s_{max},2s_1]} is the smallest scale such that
\eqn{S(s_{min})\le S(s)} for all \eqn{s \in [s_{max},2s_1]}.
}
\section{References}{


R. Benítez, V. J. Bolós, M. E. Ramírez. A wavelet-based tool for studying
non-periodicity. Comput. Math. Appl. 60 (2010), no. 3, 634-641.
}

\examples{

dt <- 0.1
time <- seq(0, 50, dt)
signal <- c(sin(pi * time), sin(pi * time / 2))
si <- scale_index(signal = signal, dt = dt)

# Another way, giving the scalogram instead of the signal:

sc <- scalogram(signal = signal, dt = dt, energy_density = FALSE, makefigure = FALSE)
si <- scale_index(scalog = sc$scalog, scales = sc$scales, dt = dt)

}
