% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/log_scores.R
\name{density_scores}
\alias{density_scores}
\alias{density_scores.default}
\alias{density_scores.kde}
\alias{density_scores.lm}
\alias{density_scores.gam}
\title{Density scores}
\usage{
density_scores(object, loo = FALSE, ...)

\method{density_scores}{default}(
  object,
  loo = FALSE,
  h = kde_bandwidth(object, method = "double"),
  H = kde_bandwidth(object, method = "double"),
  ...
)

\method{density_scores}{kde}(object, loo = FALSE, ...)

\method{density_scores}{lm}(object, loo = FALSE, ...)

\method{density_scores}{gam}(object, loo = FALSE, ...)
}
\arguments{
\item{object}{A model object or a numerical data set.}

\item{loo}{Should leave-one-out density scores be computed?}

\item{...}{Other arguments are ignored.}

\item{h}{Bandwidth for univariate kernel density estimate. Default is \code{\link{kde_bandwidth}}.}

\item{H}{Bandwidth for multivariate kernel density estimate. Default is \code{\link{kde_bandwidth}}.}
}
\value{
A numerical vector containing either the density scores, or the LOO
density scores.
}
\description{
Compute density scores or leave-one-out density scores from a
model or a kernel density estimate of a data set.
The density scores are defined as minus the log of the conditional density,
or kernel density estimate, at each observation.
The leave-one-out density scores (or LOO density scores) are obtained by
estimating the conditional density or kernel density estimate using all
other observations.
}
\details{
If the first argument is a numerical vector or matrix, then
a kernel density estimate is computed, using a Gaussian kernel,
with default bandwidth given by a robust normal reference rule.
Otherwise the model is used to compute the conditional
density function at each observation, from which the density scores (or
possibly the LOO density scores) are obtained.
}
\examples{
# Density scores computed from bivariate data set
of <- oldfaithful |>
  filter(duration < 7000, waiting < 7000) |>
  mutate(
    fscores = density_scores(cbind(duration, waiting)),
    loo_fscores = density_scores(cbind(duration, waiting), loo = TRUE),
    lookout_prob = lookout(density_scores = fscores, loo_scores = loo_fscores)
  )
of |>
  ggplot(aes(x = duration, y = waiting, color = lookout_prob < 0.01)) +
  geom_point()
# Density scores computed from bivariate KDE
f_kde <- kde(of[, 2:3], H = kde_bandwidth(of[, 2:3]))
of |>
  mutate(
    fscores = density_scores(f_kde),
    loo_fscores = density_scores(f_kde, loo = TRUE)
  )
# Density scores computed from linear model
shiraz <- wine_reviews |> filter(variety \%in\% c("Shiraz", "Syrah"))
lm_wine <- lm(log(price) ~ points, data = shiraz)
shiraz |>
  mutate(
    fscore = density_scores(lm_wine),
    loo_fscore = density_scores(lm_wine, loo = TRUE),
    lookout_prob = lookout(density_scores = fscore, loo_scores = loo_fscore)
  ) |>
  ggplot(aes(x = points, y = price, color = lookout_prob < 0.02)) +
  geom_jitter(height = 0, width = 0.2) +
  scale_y_log10()
# Density scores computed from GAM
gam_wine <- mgcv::gam(log(price) ~ s(points), data = shiraz)
shiraz |>
  mutate(
    fscore = density_scores(gam_wine),
    lookout_prob = lookout(density_scores = fscore)
  ) |>
  ggplot(aes(x = points, y = price, color = lookout_prob < 0.02)) +
  geom_jitter(height = 0, width = 0.2) +
  scale_y_log10()
}
\seealso{
\code{\link{kde_bandwidth}}
\code{\link[ks]{kde}}
}
\author{
Rob J Hyndman
}
