use super::*;
use std::any::Any;
use std::fmt::Debug;














#[derive(PartialEq, Clone)]
pub struct ExternalPtr<T: Debug + 'static> {

    pub(crate) robj: Robj,


    marker: std::marker::PhantomData<T>,
}

impl<T: Debug + 'static> robj::GetSexp for ExternalPtr<T> {
    unsafe fn get(&self) -> SEXP {
        self.robj.get()
    }


    fn as_robj(&self) -> &Robj {
        &self.robj
    }


    fn as_robj_mut(&mut self) -> &mut Robj {
        &mut self.robj
    }
}


impl<T: Debug + 'static> Length for ExternalPtr<T> {}


impl<T: Debug + 'static> Types for ExternalPtr<T> {}


impl<T: Debug + 'static> Conversions for ExternalPtr<T> {}


impl<T: Debug + 'static> Rinternals for ExternalPtr<T> {}


impl<T: Debug + 'static> Slices for ExternalPtr<T> {}


impl<T: Debug + 'static> Operators for ExternalPtr<T> {}

impl<T: Debug + 'static> Deref for ExternalPtr<T> {
    type Target = T;


    fn deref(&self) -> &Self::Target {
        self.addr()
    }
}

impl<T: Debug + 'static> DerefMut for ExternalPtr<T> {

    fn deref_mut(&mut self) -> &mut Self::Target {
        self.addr_mut()
    }
}

impl<T: Any + Debug> ExternalPtr<T> {






    pub fn new(val: T) -> Self {
        unsafe {

            let boxed = Box::new(val);



            let robj = Robj::make_external_ptr(Box::into_raw(boxed), r!(()));

            extern "C" fn finalizer<T>(x: SEXP) {
                unsafe {
                    let ptr = R_ExternalPtrAddr(x) as *mut T;


                    R_SetExternalPtrTag(x, R_NilValue);



                    drop(Box::from_raw(ptr));


                    R_ClearExternalPtr(x);
                }
            }


            robj.register_c_finalizer(Some(finalizer::<T>));


            Self {
                robj,
                marker: std::marker::PhantomData,
            }
        }
    }




    pub fn tag(&self) -> Robj {
        unsafe { Robj::from_sexp(R_ExternalPtrTag(self.robj.get())) }
    }


    pub fn protected(&self) -> Robj {
        unsafe { Robj::from_sexp(R_ExternalPtrProtected(self.robj.get())) }
    }



    pub fn addr<'a>(&self) -> &'a T {
        unsafe {
            let ptr = R_ExternalPtrAddr(self.robj.get()) as *const T;
            &*ptr as &'a T
        }
    }



    pub fn addr_mut(&mut self) -> &mut T {
        unsafe {
            let ptr = R_ExternalPtrAddr(self.robj.get()) as *mut T;
            &mut *ptr as &mut T
        }
    }
}

impl<T: Any + Debug> TryFrom<&Robj> for ExternalPtr<T> {
    type Error = Error;

    fn try_from(robj: &Robj) -> Result<Self> {
        let clone = robj.clone();
        if clone.rtype() != Rtype::ExternalPtr {
            Err(Error::ExpectedExternalPtr(clone))
        } else if clone.check_external_ptr_type::<T>() {
            let res = ExternalPtr::<T> {
                robj: clone,
                marker: std::marker::PhantomData,
            };
            Ok(res)
        } else {
            Err(Error::ExpectedExternalPtrType(
                clone,
                std::any::type_name::<T>().into(),
            ))
        }
    }
}

impl<T: Any + Debug> TryFrom<Robj> for ExternalPtr<T> {
    type Error = Error;

    fn try_from(robj: Robj) -> Result<Self> {
        <ExternalPtr<T>>::try_from(&robj)
    }
}

impl<T: Any + Debug> From<ExternalPtr<T>> for Robj {
    fn from(val: ExternalPtr<T>) -> Self {
        val.robj
    }
}

impl<T: Debug + 'static> std::fmt::Debug for ExternalPtr<T> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        (&**self as &T).fmt(f)
    }
}
