% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rarefaction.R
\name{rarefaction}
\alias{rarefaction}
\title{Rarefaction Analysis}
\usage{
rarefaction(
  formula,
  data,
  x,
  step = 1,
  points = NULL,
  prob = 0.95,
  xlab,
  ylab,
  plot = TRUE,
  theme = "theme_classic"
)
}
\arguments{
\item{formula}{An optional formula specifying the relationship between taxa and sample units
(e.g., \code{Taxon ~ Sample}). If provided, the function extracts variables from \code{data}.
A third variable may be included to remove dead individuals (e.g., \code{Taxon ~ Sample - Dead}).}

\item{data}{A data frame containing the variables specified in \code{formula} ('long format').
It must contain one column representing the sample unit labels (e.g., quadrats or points)
and one column representing the taxon names of the individual plants. This argument accepts
the data frame used in the argument \code{x} in the function \code{phytoparam}.}

\item{x}{An optional contingency table of species (rows) by samples (columns). If not provided,
it is calculated from \code{formula} and \code{data}. Alternatively, it can be a vector
representing the number of individuals per species (see Examples).}

\item{step}{Step size for the sequence of sample sizes in the rarefaction curve. Default is 1.}

\item{points}{Optional vector of specific sample sizes (breakpoints) for which to calculate rarefaction.
If \code{NULL}, a sequence from 1 to the total number of individuals is used.}

\item{prob}{The confidence level for the confidence intervals. Default is 0.95.}

\item{xlab}{Label for the x-axis of the plot (defaults to "Number of individuals").}

\item{ylab}{Label for the y-axis of the plot (defaults to "Number of species").}

\item{plot}{Logical; if \code{TRUE}, a rarefaction curve is plotted. Default is \code{TRUE}.}

\item{theme}{Character string with the name of a ggplot2 theme to be applied to the plot
(e.g., \code{"theme_light"}, \code{"theme_bw"}, \code{"theme_minimal"}).
Default is \code{"theme_classic"}.}
}
\value{
A data frame with the following components:
\itemize{
  \item \code{n}: number of sample units.
  \item \code{s}: mean number of species.
  \item \code{lower}: lower confidence interval bound.
  \item \code{upper}: upper confidence interval bound.
  }

If \code{plot = TRUE}, a rarefaction curve with confidence ribbons is produced using \pkg{ggplot2}.
}
\description{
Performs a rarefaction analysis, a method widely used in ecology to estimate species richness
based on sample size. The function computes the expected number of species for increasing numbers
of individuals, along with confidence intervals, following classical approaches by Hurlbert (1971),
Heck et al. (1975), and related developments.
}
\details{
Rarefaction analysis provides a standardized way to compare species richness among samples
of different sizes. It is based on probabilistic resampling without replacement and produces
an expected species accumulation curve. Confidence intervals are calculated following variance
estimators proposed by Heck et al. (1975) and Tipper (1979).

The function accepts data in three formats:
\itemize{
  \item long format (formula + data arguments),
  \item contingency matrix,
  \item vector of individuals per species.
}
Dead individuals can be excluded by specifying an additional term in the formula.
}
\examples{
## Using 'formula' (long format)
rarefaction(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  plot = TRUE
)

\donttest{
## Using different plot themes
rarefaction(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  plot = TRUE,
  theme = "theme_light"
)
rarefaction(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  plot = TRUE,
  theme = "theme_bw"
)
rarefaction(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  plot = TRUE,
  theme = "theme_minimal"
)

## Using a matrix (wide format)
data.matrix <- with(
  quadrat.df,
  table(Plot, Species, exclude = "Morta")
)
rarefaction(x = data.matrix, plot = TRUE)

data.matrix <- as.matrix(
  xtabs(~ Plot + Species, data = quadrat.df, exclude = "Morta")
)
rarefaction(x = data.matrix, plot = TRUE)

## Using a vector
data.vector <- sort(
  as.vector(apply(data.matrix, 2, sum)),
  decreasing = TRUE
)
rarefaction(x = data.vector, plot = TRUE)

## Using breakpoints
pts <- c(1, 10, 30, 50, 80)
rarefaction(
  formula = Species ~ Plot - Morta,
  data = quadrat.df,
  points = pts,
  plot = TRUE
)
rarefaction(x = data.matrix, points = pts, plot = TRUE)
rarefaction(
  x = data.vector,
  points = pts,
  plot = TRUE,
  theme = "theme_light"
)
rarefaction(x = data.vector, points = 50, plot = FALSE)
}
}
\references{
Colwell, R. K., Mao, C. X., & Chang, J. (2004).
Interpolating, extrapolating, and comparing incidence-based species accumulation curves.
\emph{Ecology}, 85(10), 2717--2727. \doi{10.1890/03-0557}

Heck, K. L., Van Belle, G., & Simberloff, D. (1975).
Explicit calculation of the rarefaction diversity measurement and the determination of sufficient sample size.
\emph{Ecology}, 56(6), 1459--1461. \doi{10.2307/1934716}

Hurlbert, S. H. (1971).
The nonconcept of species diversity: A critique and alternative parameters.
\emph{Ecology}, 52(4), 577--586. \doi{10.2307/1934145}

Tipper, J. C. (1979).
Rarefaction and rarefiction---The use and abuse of a method in paleoecology.
\emph{Paleobiology}, 5(4), 423--434. \doi{10.1017/S0094837300016924}
}
\author{
Rodrigo Augusto Santinelo Pereira \email{raspereira@usp.br}
}
