% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integrateGammaLines.R
\name{integrateGammaLines}
\alias{integrateGammaLines}
\title{Integrate Gamma line surveys}
\usage{
integrateGammaLines(
  object,
  newdata = NULL,
  w.lo = NULL,
  w.hi = NULL,
  Units = NULL
)
}
\arguments{
\item{object}{Either an Rdistance fitted distance function
(an object that inherits from class "dfunc"; usually produced 
by a call to \code{\link{dfuncEstim}}), or a matrix of canonical 
distance function parameters (e.g., \code{matrix(exp(fit$par),1)}). 
If a matrix, each row corresponds to a 
distance function and each column is a parameter. The first column is 
the parameter related to sighting covariates and must be transformed 
to the "real" space (i.e., inverse link, which is \eqn{exp()}, must 
be applied outside this routine).  If \code{object} is a matrix, 
it should not have measurement units because
only derived quantities (e.g., ESW) have units; Rdistance function 
parameters themselves never have units.}

\item{newdata}{A data frame containing new values for 
the distance function covariates. If NULL and 
\code{object} is a fitted distance function, the  
observed covariates stored in
\code{object} are used (behavior similar to \code{\link{predict.lm}}).
Argument \code{newdata} is ignored if \code{object} is a matrix.}

\item{w.lo}{Minimum sighting distance or left-truncation value
if \code{object} is a matrix.
Ignored if \code{object} 
is a fitted distance function. 
Must have physical measurement units.}

\item{w.hi}{Maximum sighting distance or right-truncation value
if \code{object} is a matrix.
Ignored if \code{object} 
is a fitted distance function.
Must have physical measurement units.}

\item{Units}{Physical units of sighting distances if 
\code{object} is a matrix. Sighting distance units can differ from units 
of \code{w.lo} or \code{w.hi}.   Ignored if \code{object}
is a fitted distance function.}
}
\value{
A vector of areas under the distance functions represented in 
\code{object}. 
If \code{object} is a distance function and 
\code{newdata} is specified, the returned vector's length is 
\code{nrow(newdata)}.  If \code{object} is a distance function and 
\code{newdata} is NULL, 
returned vector's length is \code{length(distances(object))}. If 
\code{object} is a matrix, return's length is 
\code{nrow(object)}.
}
\description{
Compute integral of the Gamma distance function for 
line-transect surveys.
}
\details{
#' Returned integrals are
\deqn{\int_0^{w} \left(\frac{x}{m}\right)^{\alpha -1}  e^{-(x - m)/\sigma_i}dx,}{
Integral( (x/m)^(a-1) exp(-(x-m)/s_i) ),}
where \eqn{w = w.hi - w.lo}, \eqn{\sigma_i}{s_i} is the i-th estimated scale 
parameter for the Gamma distance function, and \eqn{m} is the mode of Gamma
(i.e., \eqn{(\alpha - 1)\sigma_i}{(a-1)s_i}. 
Rdistance computes the integral using R's base function 
\code{pgamma()}, which for all intents and purposes is exact. 
See also \code{\link{Gamma.like}}.
}
\section{Note}{

Users will not normally call this function. It is called 
internally by \code{\link{nLL}} and \code{\link{effectiveDistance}}.
}

\examples{

# Fake distance function object w/ minimum inputs for integration
d <- rep(1,4) \%m\%. # Only units needed, not values
obs <- factor(rep(c("obs1", "obs2"), 2))
beta <- c(4.0, -0.5, 1.5) # {'Intercept', b_1, shape}
w.hi <- 125
w.lo <- 20
ml <- list(
    mf = model.frame(d ~ obs)
  , par = beta 
  , likelihood = "Gamma"
  , w.lo = w.lo \%#\% "m"
  , w.hi = w.hi \%#\% "m"
  , expansions = 0
)
class(ml) <- "dfunc"
integrateGammaLines(ml)

# Check: Integral of Gamma density from 0 to w.hi-w.lo
b <- exp(c(beta[1], beta[1] + beta[2]))
B <- Rdistance::GammaReparam(shp = beta[3], scl = b)
m <- (B$shp - 1)*B$scl
f.at.m <- dgamma(m, shape = B$shp, scale = B$scl)
intgral <- pgamma(q = w.hi - w.lo, shape = B$shp, scale = B$scl) / f.at.m
intgral


}
\seealso{
\code{\link{integrateNumeric}}; \code{\link{integrateNegexpLines}}; 
\code{\link{integrateOneStepLines}}
}
