# functions for calculating approximate confidence regions for eigenvalues in a fixed-trace situation

# For an ellipse size a and b on the x and y axis, respectively
# return the (x,y) location corresponding to the ray at the angle intersecting the unit circle, then scaled to have the correct a or b
# that is gives the (x,y) such that (x/a, y/b) (which is on the unit circle) corresponds to the given angle
# angle may be a vector of angles
# @param angle Angle of ray relative to x-axis
# @param a Distance from origin to ellipse along the x-axis
# @param b Distance from origin to ellipse along the y-axis
regularellipse <- function(angle, a, b){
  x <- a * cos(angle)
  y <- b * sin(angle)
  return(cbind(x = x, y = y))
}

# convert points from regular ellipse to the fixed trace plane through the origin with rotation by eigenvectors of Omega
# @param pts from regularellipse()
# @param evecs Eigenvectors of Omega
ellipseinftplane <- function(pts, evecs){
  rotpts <- evecs %*% t(pts)
  Hfull <- helmert(ncol(pts) + 1)
  cbind(0, t(rotpts)) %*% Hfull
}

# combine ellipseinftplane and regular ellipse to get locations around the mean
# also shifts the ellipse into the plane through ctrevals
# @param ctrevals Location of the center of the ellipse
ellipseftcentre <- function(angle, a, b, evecs, ctrevals){
  locs2d <- regularellipse(angle, a = a, b = b)
  locsplane <- ellipseinftplane(locs2d, evecs = evecs)
  locsplanearoundcenter <- t(ctrevals - t(locsplane))
  return(locsplanearoundcenter)
}

#' @title Eigenvalue confidence region under fixed trace constraint
#' @description When a 3x3 symmetric matrix has a fixed-trace constraint, the vector of its eigenvalues lies on a 2D plane.
#' This function calculates the boundary of an approximate confidence region in this 2D plane using the same statistic as [`test_fixedtrace()`].
#' The returned boundary can be used to plot the confidence region.
#' The function [`conf_fixedtrace_inregion()`] returns where given points are in the estimated confidence region.
#' @details
#' Uses the same statistic as [`test_fixedtrace()`] and bootstrap resampling to obtain approximate bounds on the eigenvalues of a population mean.
#' The statistic has a quadratic form so that the boundary of the confidence region is an ellipse, but for plotting simplicity the ellipse is returned as a dense set of `npts` points.
#' A warning will be generated if the confidence region leaves the space of distinct descending-order eigenvalues and a check of coverage of bootstrap resamples is available.
#' @param x A single sample of 3x3 symmetric matrices. `x` must be either an [`fsm`] object or something that [`as_fsm()`] can parse.
#' @param alpha Desired significance level of the approximate confidence region.
#' @param B Number of bootstrap resamples.
#' @param npts Number of points on the boundary of the region to compute.
#' @param check If `TRUE`, then the extrinsic means of 100 new resamples will be used to check the coverage of the region.
#' @return A list:
#' + `est`: the eigenvalues of the mean matrix.
#' + `boundary`: A matrix with 3 columns and `npts` rows giving the boundary of the region. Each row corresponds to a point on the boundary and the columns are the first, second and final eigenvalue.
#' + `Omega`: The estimated covariance of the (projected) eigenvalues
#' + `threshold`: The threshold (estimated via resampling) on the statistic.
#' @export
conf_fixedtrace <- function(x, alpha = 0.05, B = 1000, npts = 1000, check = TRUE){
  x <- as_fsm(x)
  stopifnot(ncol(x) == 6) #ensures input is 3x3 symmetric matrix data
  stopifnot(has_fixedtrace(x))

  # sample mean
  av <- mmean(x)
  av_ess <- eigen_desc(av)
  av_eval <- av_ess$values
  size <- nrow(x)

  # resampling and computing stat_fixedtrace()
  res <- boot_calib(x, x, stat = stat_fixedtrace, B = B, evals = av_eval)
  statthreshold <- stats::quantile(res$nullt, probs = 1-alpha, names = FALSE, type = 1)

  # now compute boundary of region
  # solves for the set of locations such that stat_fixedtrace() is smaller than the statthreshold
  Omega <- cov_evals_ft(x, evecs = av_ess$vectors, av = av)
  Omega_ess <- eigen_desc(Omega)
  a <- sqrt(statthreshold * Omega_ess$values[1]/size)
  b <- sqrt(statthreshold * Omega_ess$values[2]/size)
  stopifnot(all(Omega_ess$values >= 0))
  bdrypts <- ellipseftcentre(angle = seq(0, 2*pi, length.out = npts),
                         a = a,
                         b = b,
                         evecs = Omega_ess$vectors,
                         ctrevals = av_eval
                         )

  # also create a function that tests whether inside the region using the statistic directly
  # this is very similar to conf_fixedtrace_inregion() below, but faster because Omega_ess only need to be calculated once here
  H <- helmertsub(3)
  inregion <- function(evals){
    statval <- size * t(av_eval - evals) %*% t(H) %*% Omega_ess$vectors %*% diag(1/Omega_ess$values) %*% t(Omega_ess$vectors) %*% H %*% (av_eval - evals)
    return(drop(statval) <= statthreshold)
  }

  # now check if close to the descending order boundary by solving for the intersection of the boundary lines and the ellipse
  # the sqrt(6) comes from the definition of the helmertsub matrix
  # see ConfidenceRegionsFT.pdf EQ1 and EQ2 for obtaining the equations
  # first two evals:
  ft <- sum(av_eval)
  A <- t(Omega_ess$vectors) %*% (H %*% av_eval + (2*ft/sqrt(6)) * rbind(0, 1))
  B <- sqrt(6)  * t(Omega_ess$vectors)  %*%  rbind(0, 1)
  # b^2 - 4ac >= 0
  invLambda <- diag(1/Omega_ess$values)
  e1e2intersect <- drop(
  (t(A) %*% invLambda %*% B + t(B) %*%  invLambda %*% A)^2 -
    4 * (t(B) %*% invLambda %*% B) * ((t(A) %*% invLambda %*% A) - statthreshold/size)
  ) >= 0
  
  # second two evals
  A <- t(Omega_ess$vectors) %*% (H %*% av_eval + (ft/2) * rbind(1/sqrt(2), 1/sqrt(6)))
  B <- (3/2) * t(Omega_ess$vectors) %*% rbind(1/sqrt(2), 1/sqrt(6))
  e2e3intersect <- drop(
    (t(A) %*% invLambda %*% B + t(B) %*%  invLambda %*% A)^2 -
      4 * (t(B) %*% invLambda %*% B) * ((t(A) %*% invLambda %*% A) - statthreshold/size)
  ) >= 0
  
  
  if (e1e2intersect | e2e3intersect){
    desc <- switch(e1e2intersect + 2 * e2e3intersect,
           "largest two eigenvalues",
           "smallest two eigenvalues",
           "largest two eigenvalues or smallest two eigenvalues")
    warning(sprintf("Confidence region includes eigenvalues where the %s are not in descending order.", desc))
  } 
  
  if (check){ # the region should contain 1-alpha of mean of resampled data, this does a quick check
    resample_avevals <- t(replicate(100, eigen_desc(mmean(sample_fsm(x)))$values))
    inregionvals <- apply(resample_avevals, MARGIN = 1, function(v) {inregion(v)})
    coverage <- mean(inregionvals)
    coverage_sd <- stats::sd(inregionvals)/sqrt(length(inregionvals))
    if (coverage + 2 * coverage_sd < 1-alpha){
      warning(sprintf("Interval covers only %0.0f%% of resample means.", coverage * 100))
    }
    if (coverage - 2 * coverage_sd > 1-alpha){
      warning(sprintf("Interval covers %0.0f%% of resample means.", coverage * 100))
    }
  }

  return(list(
    est = av_eval,
    boundary = bdrypts,
    Omega = Omega,
    threshold = statthreshold,
    bootstat = res$nullt,
    samplesize = size
  ))
}

# @describeIn conf_fixedtrace Return whether a particular set of eigenvalues `evals` lies in the confidence region returned by [conf_fixedtrace()].
#' @rdname conf_fixedtrace
#' @param evals A set of eigenvalues with the same trace as matrices in `x`.
#' @param cr A confidence region returned by [conf_fixedtrace()].
#' @export
conf_fixedtrace_inregion <- function(evals, cr){
  H <- helmertsub(length(cr$est))
  Omega_ess <- eigen_desc(cr$Omega)
  statval <- cr$samplesize * t(cr$est - evals) %*% t(H) %*% Omega_ess$vectors %*% diag(1/Omega_ess$values) %*% t(Omega_ess$vectors) %*% H %*% (cr$est - evals)
  return(drop(statval) <= cr$threshold)
}
