/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5B2module.h" 

#include "H5private.h"   
#include "H5B2pkg.h"     
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 

static herr_t H5B2__shadow_leaf(H5B2_leaf_t *leaf, H5B2_node_ptr_t *curr_node_ptr);

H5FL_DEFINE(H5B2_leaf_t);

herr_t
H5B2__create_leaf(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr)
{
    H5B2_leaf_t *leaf      = NULL;    
    bool         inserted  = false;   
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(node_ptr);

    
    if (NULL == (leaf = H5FL_CALLOC(H5B2_leaf_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for B-tree leaf info");

    
    if (H5B2__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINC, FAIL, "can't increment ref. count on B-tree header");

    
    leaf->hdr = hdr;

    
    if (NULL == (leaf->leaf_native = (uint8_t *)H5FL_FAC_MALLOC(hdr->node_info[0].nat_rec_fac)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for B-tree leaf native keys");
    memset(leaf->leaf_native, 0, hdr->cls->nrec_size * hdr->node_info[0].max_nrec);

    
    leaf->parent = parent;

    
    leaf->shadow_epoch = hdr->shadow_epoch;

    
    if (HADDR_UNDEF == (node_ptr->addr = H5MF_alloc(hdr->f, H5FD_MEM_BTREE, (hsize_t)hdr->node_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "file allocation failed for B-tree leaf node");

    
    if (H5AC_insert_entry(hdr->f, H5AC_BT2_LEAF, node_ptr->addr, leaf, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL, "can't add B-tree leaf to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, leaf) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, FAIL, "unable to add v2 B-tree node as child of proxy");
        leaf->top_proxy = hdr->top_proxy;
    } 

done:
    if (ret_value < 0) {
        if (leaf) {
            
            if (inserted)
                if (H5AC_remove_entry(leaf) < 0)
                    HDONE_ERROR(H5E_BTREE, H5E_CANTREMOVE, FAIL,
                                "unable to remove v2 B-tree leaf node from cache");

            
            if (H5_addr_defined(node_ptr->addr) &&
                H5MF_xfree(hdr->f, H5FD_MEM_BTREE, node_ptr->addr, (hsize_t)hdr->node_size) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL,
                            "unable to release file space for v2 B-tree leaf node");

            
            if (H5B2__leaf_free(leaf) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to release v2 B-tree leaf node");
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5B2_leaf_t *
H5B2__protect_leaf(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr, bool shadow, unsigned flags)
{
    H5B2_leaf_cache_ud_t udata;            
    H5B2_leaf_t         *leaf;             
    H5B2_leaf_t         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(node_ptr);
    assert(H5_addr_defined(node_ptr->addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.f      = hdr->f;
    udata.hdr    = hdr;
    udata.parent = parent;
    udata.nrec   = node_ptr->node_nrec;

    
    if (NULL == (leaf = (H5B2_leaf_t *)H5AC_protect(hdr->f, H5AC_BT2_LEAF, node_ptr->addr, &udata, flags)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, NULL, "unable to protect B-tree leaf node");

    
    if (hdr->top_proxy && NULL == leaf->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, leaf) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, NULL, "unable to add v2 B-tree leaf node as child of proxy");
        leaf->top_proxy = hdr->top_proxy;
    } 

    
    if (shadow)
        if (H5B2__shadow_leaf(leaf, node_ptr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, NULL, "unable to shadow leaf node");

    
    ret_value = leaf;

done:
    
    if (!ret_value) {
        
        if (leaf) {
            
            if (leaf->top_proxy) {
                if (H5AC_proxy_entry_remove_child(leaf->top_proxy, leaf) < 0)
                    HDONE_ERROR(
                        H5E_BTREE, H5E_CANTUNDEPEND, NULL,
                        "unable to destroy flush dependency between leaf node and v2 B-tree 'top' proxy");
                leaf->top_proxy = NULL;
            } 

            
            if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, node_ptr->addr, leaf, H5AC__NO_FLAGS_SET) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, NULL,
                            "unable to unprotect v2 B-tree leaf node, address = %llu",
                            (unsigned long long)node_ptr->addr);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__neighbor_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, void *neighbor_loc, H5B2_compare_t comp,
                    void *parent, void *udata, H5B2_found_t op, void *op_data)
{
    H5B2_leaf_t *leaf;                
    unsigned     idx       = 0;       
    int          cmp       = 0;       
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));
    assert(op);

    
    if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, curr_node_ptr, false, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

    
    if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
    if (cmp > 0)
        idx++;
    else if (cmp == 0 && comp == H5B2_COMPARE_GREATER)
        idx++;

    
    if (comp == H5B2_COMPARE_LESS) {
        if (idx > 0)
            neighbor_loc = H5B2_LEAF_NREC(leaf, hdr, idx - 1);
    } 
    else {
        assert(comp == H5B2_COMPARE_GREATER);

        if (idx < leaf->nrec)
            neighbor_loc = H5B2_LEAF_NREC(leaf, hdr, idx);
    } 

    
    if (neighbor_loc) {
        
        if ((op)(neighbor_loc, op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL,
                        "'found' callback failed for B-tree neighbor operation");
    } 
    else
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "unable to find neighbor record in B-tree");

done:
    
    if (leaf && H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr->addr, leaf, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree leaf node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__insert_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos, void *parent,
                  void *udata)
{
    H5B2_leaf_t *leaf;                            
    unsigned     leaf_flags = H5AC__NO_FLAGS_SET; 
    int          cmp;                             
    unsigned     idx       = 0;                   
    herr_t       ret_value = SUCCEED;             

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, curr_node_ptr, false, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

    
    assert(curr_node_ptr->node_nrec < hdr->node_info[0].max_nrec);

    
    assert(curr_node_ptr->all_nrec == curr_node_ptr->node_nrec);
    assert(leaf->nrec == curr_node_ptr->node_nrec);

    
    if (leaf->nrec == 0)
        idx = 0;
    else {
        
        if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        if (cmp == 0)
            HGOTO_ERROR(H5E_BTREE, H5E_EXISTS, FAIL, "record is already in B-tree");
        if (cmp > 0)
            idx++;

        
        if (idx < leaf->nrec)
            memmove(H5B2_LEAF_NREC(leaf, hdr, idx + 1), H5B2_LEAF_NREC(leaf, hdr, idx),
                    hdr->cls->nrec_size * (leaf->nrec - idx));
    } 

    
    if ((hdr->cls->store)(H5B2_LEAF_NREC(leaf, hdr, idx), udata) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert record into leaf node");

    
    leaf_flags |= H5AC__DIRTIED_FLAG;

    
    curr_node_ptr->all_nrec++;
    curr_node_ptr->node_nrec++;

    
    leaf->nrec++;

    
    
    if (H5B2_POS_MIDDLE != curr_pos) {
        if (idx == 0) {
            if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->min_native_rec == NULL)
                    if (NULL == (hdr->min_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for v2 B-tree min record info");
                H5MM_memcpy(hdr->min_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
            } 
        }     
        if (idx == (unsigned)(leaf->nrec - 1)) {
            if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->max_native_rec == NULL)
                    if (NULL == (hdr->max_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for v2 B-tree max record info");
                H5MM_memcpy(hdr->max_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
            } 
        }     
    }         

done:
    
    if (leaf) {
        
        if (hdr->swmr_write && (leaf_flags & H5AC__DIRTIED_FLAG))
            if (H5B2__shadow_leaf(leaf, curr_node_ptr) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow leaf B-tree node");

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr->addr, leaf, leaf_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release leaf B-tree node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__update_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_update_status_t *status,
                  H5B2_nodepos_t curr_pos, void *parent, void *udata, H5B2_modify_t op, void *op_data)
{
    H5B2_leaf_t *leaf;                            
    unsigned     leaf_flags = H5AC__NO_FLAGS_SET; 
    int          cmp        = -1;                 
    unsigned     idx        = 0;                  
    herr_t       ret_value  = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, curr_node_ptr, false, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");

    
    assert(curr_node_ptr->all_nrec == curr_node_ptr->node_nrec);
    assert(leaf->nrec == curr_node_ptr->node_nrec);

    
    if (leaf->nrec == 0)
        idx = 0;
    else {
        
        if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");

        
        if (0 != cmp) {
            
            if (curr_node_ptr->node_nrec == hdr->node_info[0].split_nrec) {
                
                *status = H5B2_UPDATE_INSERT_CHILD_FULL;

                
                HGOTO_DONE(SUCCEED);
            } 

            
            if (cmp > 0)
                idx++;

            
            if (idx < leaf->nrec)
                memmove(H5B2_LEAF_NREC(leaf, hdr, idx + 1), H5B2_LEAF_NREC(leaf, hdr, idx),
                        hdr->cls->nrec_size * (leaf->nrec - idx));
        } 
    }     

    
    if (0 == cmp) {
        bool changed = false; 

        
        if ((op)(H5B2_LEAF_NREC(leaf, hdr, idx), op_data, &changed) < 0) {
            
            assert(changed == false);

            HGOTO_ERROR(H5E_BTREE, H5E_CANTMODIFY, FAIL,
                        "'modify' callback failed for B-tree update operation");
        } 

        
        leaf_flags |= (changed ? H5AC__DIRTIED_FLAG : 0);

        
        *status = H5B2_UPDATE_MODIFY_DONE;
    } 
    else {
        
        assert(curr_node_ptr->node_nrec < hdr->node_info[0].max_nrec);

        
        if ((hdr->cls->store)(H5B2_LEAF_NREC(leaf, hdr, idx), udata) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, FAIL, "unable to insert record into leaf node");

        
        leaf_flags |= H5AC__DIRTIED_FLAG;

        
        *status = H5B2_UPDATE_INSERT_DONE;

        
        curr_node_ptr->all_nrec++;
        curr_node_ptr->node_nrec++;

        
        leaf->nrec++;
    } 

    
    
    if (H5B2_POS_MIDDLE != curr_pos) {
        if (idx == 0) {
            if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->min_native_rec == NULL)
                    if (NULL == (hdr->min_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for v2 B-tree min record info");
                H5MM_memcpy(hdr->min_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
            } 
        }     
        if (idx == (unsigned)(leaf->nrec - 1)) {
            if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->max_native_rec == NULL)
                    if (NULL == (hdr->max_native_rec = H5MM_malloc(hdr->cls->nrec_size)))
                        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for v2 B-tree max record info");
                H5MM_memcpy(hdr->max_native_rec, H5B2_LEAF_NREC(leaf, hdr, idx), hdr->cls->nrec_size);
            } 
        }     
    }         

done:
    
    if (leaf) {
        
        if (hdr->swmr_write && (leaf_flags & H5AC__DIRTIED_FLAG)) {
            
            if (H5B2__shadow_leaf(leaf, curr_node_ptr) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow leaf B-tree node");

            
            
            if (*status == H5B2_UPDATE_MODIFY_DONE)
                *status = H5B2_UPDATE_SHADOW_DONE;
        } 

        
        if (H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, curr_node_ptr->addr, leaf, leaf_flags) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release leaf B-tree node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__swap_leaf(H5B2_hdr_t *hdr, uint16_t depth, H5B2_internal_t *internal, unsigned *internal_flags_ptr,
                unsigned idx, void *swap_loc)
{
    const H5AC_class_t *child_class;              
    haddr_t             child_addr = HADDR_UNDEF; 
    void               *child      = NULL;        
    uint8_t            *child_native;             
    herr_t              ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(internal);
    assert(internal_flags_ptr);
    assert(idx <= internal->nrec);

    
    if (depth > 1) {
        H5B2_internal_t *child_internal; 

        
        child_class = H5AC_BT2_INT;

        
        if (NULL ==
            (child_internal = H5B2__protect_internal(hdr, internal, &internal->node_ptrs[idx],
                                                     (uint16_t)(depth - 1), false, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree internal node");
        child_addr = internal->node_ptrs[idx].addr;

        
        child        = child_internal;
        child_native = child_internal->int_native;
    } 
    else {
        H5B2_leaf_t *child_leaf; 

        
        child_class = H5AC_BT2_LEAF;

        
        if (NULL == (child_leaf = H5B2__protect_leaf(hdr, internal, &internal->node_ptrs[idx], false,
                                                     H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");
        child_addr = internal->node_ptrs[idx].addr;

        
        child        = child_leaf;
        child_native = child_leaf->leaf_native;
    } 

    
    H5MM_memcpy(hdr->page, H5B2_NAT_NREC(child_native, hdr, 0), hdr->cls->nrec_size);
    H5MM_memcpy(H5B2_NAT_NREC(child_native, hdr, 0), swap_loc, hdr->cls->nrec_size);
    H5MM_memcpy(swap_loc, hdr->page, hdr->cls->nrec_size);

    
    *internal_flags_ptr |= H5AC__DIRTIED_FLAG;

#ifdef H5B2_DEBUG
    H5B2__assert_internal((hsize_t)0, hdr, internal);
    if (depth > 1)
        H5B2__assert_internal(internal->node_ptrs[idx].all_nrec, hdr, (H5B2_internal_t *)child);
    else
        H5B2__assert_leaf(hdr, (H5B2_leaf_t *)child);
#endif 

done:
    
    if (child && H5AC_unprotect(hdr->f, child_class, child_addr, child, H5AC__DIRTIED_FLAG) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release B-tree child node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B2__shadow_leaf(H5B2_leaf_t *leaf, H5B2_node_ptr_t *curr_node_ptr)
{
    H5B2_hdr_t *hdr;                 
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(leaf);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));
    hdr = leaf->hdr;
    assert(hdr);
    assert(hdr->swmr_write);

    
    if (leaf->shadow_epoch <= hdr->shadow_epoch) {
        haddr_t new_node_addr; 

        
        
        if (HADDR_UNDEF == (new_node_addr = H5MF_alloc(hdr->f, H5FD_MEM_BTREE, (hsize_t)hdr->node_size)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, FAIL, "unable to allocate file space to move B-tree node");

        
        if (H5AC_move_entry(hdr->f, H5AC_BT2_LEAF, curr_node_ptr->addr, new_node_addr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTMOVE, FAIL, "unable to move B-tree node");
        curr_node_ptr->addr = new_node_addr;

        

        
        leaf->shadow_epoch = hdr->shadow_epoch + 1;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__remove_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos, void *parent,
                  void *udata, H5B2_remove_t op, void *op_data)
{
    H5B2_leaf_t *leaf;                            
    haddr_t      leaf_addr  = HADDR_UNDEF;        
    unsigned     leaf_flags = H5AC__NO_FLAGS_SET; 
    unsigned     idx        = 0;                  
    int          cmp;                             
    herr_t       ret_value = SUCCEED;             

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, curr_node_ptr, false, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");
    leaf_addr = curr_node_ptr->addr;

    
    assert(curr_node_ptr->all_nrec == curr_node_ptr->node_nrec);
    assert(leaf->nrec == curr_node_ptr->node_nrec);

    
    if (H5B2__locate_record(hdr->cls, leaf->nrec, hdr->nat_off, leaf->leaf_native, udata, &idx, &cmp) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
    if (cmp != 0)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "record is not in B-tree");

    
    if (H5B2_POS_MIDDLE != curr_pos) {
        
        if (idx == 0) {
            if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->min_native_rec)
                    hdr->min_native_rec = H5MM_xfree(hdr->min_native_rec);
            } 
        }     
        if (idx == (unsigned)(leaf->nrec - 1)) {
            if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->max_native_rec)
                    hdr->max_native_rec = H5MM_xfree(hdr->max_native_rec);
            } 
        }     
    }         

    
    if (op)
        if ((op)(H5B2_LEAF_NREC(leaf, hdr, idx), op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record into leaf node");

    
    leaf->nrec--;

    if (leaf->nrec > 0) {
        
        if (hdr->swmr_write) {
            if (H5B2__shadow_leaf(leaf, curr_node_ptr) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow leaf node");
            leaf_addr = curr_node_ptr->addr;
        } 

        
        if (idx < leaf->nrec)
            memmove(H5B2_LEAF_NREC(leaf, hdr, idx), H5B2_LEAF_NREC(leaf, hdr, (idx + 1)),
                    hdr->cls->nrec_size * (leaf->nrec - idx));

        
        leaf_flags |= H5AC__DIRTIED_FLAG;
    } 
    else {
        
        leaf_flags |= H5AC__DELETED_FLAG;
        if (!hdr->swmr_write)
            leaf_flags |= H5AC__DIRTIED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

        
        curr_node_ptr->addr = HADDR_UNDEF;
    } 

    
    curr_node_ptr->node_nrec--;

done:
    
    if (leaf && H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, leaf_addr, leaf, leaf_flags) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release leaf B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__remove_leaf_by_idx(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos,
                         void *parent, unsigned idx, H5B2_remove_t op, void *op_data)
{
    H5B2_leaf_t *leaf;                            
    haddr_t      leaf_addr  = HADDR_UNDEF;        
    unsigned     leaf_flags = H5AC__NO_FLAGS_SET; 
    herr_t       ret_value  = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(curr_node_ptr);
    assert(H5_addr_defined(curr_node_ptr->addr));

    
    if (NULL == (leaf = H5B2__protect_leaf(hdr, parent, curr_node_ptr, false, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, FAIL, "unable to protect B-tree leaf node");
    leaf_addr = curr_node_ptr->addr;

    
    assert(curr_node_ptr->all_nrec == curr_node_ptr->node_nrec);
    assert(leaf->nrec == curr_node_ptr->node_nrec);
    assert(idx < leaf->nrec);

    
    if (H5B2_POS_MIDDLE != curr_pos) {
        
        if (idx == 0) {
            if (H5B2_POS_LEFT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->min_native_rec)
                    hdr->min_native_rec = H5MM_xfree(hdr->min_native_rec);
            } 
        }     
        if (idx == (unsigned)(leaf->nrec - 1)) {
            if (H5B2_POS_RIGHT == curr_pos || H5B2_POS_ROOT == curr_pos) {
                if (hdr->max_native_rec)
                    hdr->max_native_rec = H5MM_xfree(hdr->max_native_rec);
            } 
        }     
    }         

    
    if (op)
        if ((op)(H5B2_LEAF_NREC(leaf, hdr, idx), op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to remove record into leaf node");

    
    leaf->nrec--;

    if (leaf->nrec > 0) {
        
        if (hdr->swmr_write) {
            if (H5B2__shadow_leaf(leaf, curr_node_ptr) < 0)
                HGOTO_ERROR(H5E_BTREE, H5E_CANTCOPY, FAIL, "unable to shadow leaf node");
            leaf_addr = curr_node_ptr->addr;
        } 

        
        if (idx < leaf->nrec)
            memmove(H5B2_LEAF_NREC(leaf, hdr, idx), H5B2_LEAF_NREC(leaf, hdr, (idx + 1)),
                    hdr->cls->nrec_size * (leaf->nrec - idx));

        
        leaf_flags |= H5AC__DIRTIED_FLAG;
    } 
    else {
        
        leaf_flags |= H5AC__DELETED_FLAG;
        if (!hdr->swmr_write)
            leaf_flags |= H5AC__DIRTIED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

        
        curr_node_ptr->addr = HADDR_UNDEF;
    } 

    
    curr_node_ptr->node_nrec--;

done:
    
    if (leaf && H5AC_unprotect(hdr->f, H5AC_BT2_LEAF, leaf_addr, leaf, leaf_flags) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release leaf B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__leaf_free(H5B2_leaf_t *leaf)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(leaf);

    
    if (leaf->leaf_native)
        leaf->leaf_native = (uint8_t *)H5FL_FAC_FREE(leaf->hdr->node_info[0].nat_rec_fac, leaf->leaf_native);

    
    if (H5B2__hdr_decr(leaf->hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTDEC, FAIL, "can't decrement ref. count on B-tree header");

    
    assert(NULL == leaf->top_proxy);

    
    leaf = H5FL_FREE(H5B2_leaf_t, leaf);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5B2_DEBUG

H5_ATTR_PURE herr_t
H5B2__assert_leaf(const H5B2_hdr_t H5_ATTR_NDEBUG_UNUSED *hdr, const H5B2_leaf_t H5_ATTR_NDEBUG_UNUSED *leaf)
{
    
    assert(leaf->nrec <= hdr->node_info->split_nrec);

    return (0);
} 

H5_ATTR_PURE herr_t
H5B2__assert_leaf2(const H5B2_hdr_t H5_ATTR_NDEBUG_UNUSED *hdr, const H5B2_leaf_t H5_ATTR_NDEBUG_UNUSED *leaf,
                   const H5B2_leaf_t H5_ATTR_UNUSED *leaf2)
{
    
    assert(leaf->nrec <= hdr->node_info->split_nrec);

    return (0);
} 
#endif 
