/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Emodule.h" 

#include "H5private.h"   
#include "H5Epkg.h"      
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 

#ifndef H5_NO_DEPRECATED_SYMBOLS

char *
H5Eget_major(H5E_major_t maj)
{
    H5E_msg_t *msg; 
    ssize_t    size;
    H5E_type_t type;
    char      *msg_str = NULL;
    char      *ret_value; 

    FUNC_ENTER_API_NOCLEAR(NULL)

    
    if (NULL == (msg = (H5E_msg_t *)H5I_object_verify(maj, H5I_ERROR_MSG)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a error message ID");

    
    if ((size = H5E__get_msg(msg, &type, NULL, (size_t)0)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "can't get error message text");
    if (type != H5E_MAJOR)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "Error message isn't a major one");

    
    size++;
    msg_str = (char *)H5MM_malloc((size_t)size);

    
    if (H5E__get_msg(msg, NULL, msg_str, (size_t)size) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "can't get error message text");

    ret_value = msg_str;

done:
    if (!ret_value)
        msg_str = (char *)H5MM_xfree(msg_str);

    FUNC_LEAVE_API(ret_value)
} 

char *
H5Eget_minor(H5E_minor_t min)
{
    H5E_msg_t *msg; 
    ssize_t    size;
    H5E_type_t type;
    char      *msg_str = NULL;
    char      *ret_value; 

    FUNC_ENTER_API_NOCLEAR(NULL)

    
    if (NULL == (msg = (H5E_msg_t *)H5I_object_verify(min, H5I_ERROR_MSG)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a error message ID");

    
    if ((size = H5E__get_msg(msg, &type, NULL, (size_t)0)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "can't get error message text");
    if (type != H5E_MINOR)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "Error message isn't a minor one");

    
    size++;
    msg_str = (char *)H5MM_malloc((size_t)size);

    
    if (H5E__get_msg(msg, NULL, msg_str, (size_t)size) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, NULL, "can't get error message text");

    ret_value = msg_str;

done:
    if (!ret_value)
        msg_str = (char *)H5MM_xfree(msg_str);

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Epush1(const char *file, const char *func, unsigned line, H5E_major_t maj, H5E_minor_t min, const char *str)
{
    H5E_stack_t *estack;              
    const char  *tmp_file;            
    const char  *tmp_func;            
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get the default error stack");

    
    if (!estack->paused) {
        
        if (NULL == (tmp_file = strdup(file)))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTALLOC, FAIL, "can't duplicate file string");
        if (NULL == (tmp_func = strdup(func)))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTALLOC, FAIL, "can't duplicate function string");

        
        if (maj < H5E_first_maj_id_g || maj > H5E_last_maj_id_g)
            if (H5I_inc_ref(maj, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "can't increment major error ID");
        if (min < H5E_first_min_id_g || min > H5E_last_min_id_g)
            if (H5I_inc_ref(min, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "can't increment minor error ID");

        
        if (H5E__push_stack(estack, true, tmp_file, tmp_func, line, H5E_ERR_CLS_g, maj, min, str, NULL) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't push error on stack");
    }

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eclear1(void)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (H5E_clear_stack() < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't clear error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eprint1(FILE *stream)
{
    H5E_stack_t *estack;              
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    if (H5E__print(estack, stream, true) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTLIST, FAIL, "can't display error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ewalk1(H5E_direction_t direction, H5E_walk1_t func, void *client_data)
{
    H5E_stack_t  *estack;              
    H5E_walk_op_t walk_op;             
    herr_t        ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    walk_op.vers    = 1;
    walk_op.u.func1 = func;
    if (H5E__walk(estack, direction, &walk_op, client_data) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTLIST, FAIL, "can't walk error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eget_auto1(H5E_auto1_t *func , void **client_data )
{
    H5E_stack_t  *estack;              
    H5E_auto_op_t auto_op;             
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    if (H5E__get_auto(estack, &auto_op, client_data) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get automatic error info");

    
    if (!auto_op.is_default && auto_op.vers == 2)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "wrong API function, H5Eset_auto2 has been called");

    if (func)
        *func = auto_op.func1;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eset_auto1(H5E_auto1_t func, void *client_data)
{
    H5E_stack_t  *estack;              
    H5E_auto_op_t auto_op;             
    herr_t        ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    if (NULL == (estack = H5E__get_my_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");

    
    if (H5E__get_auto(estack, &auto_op, NULL) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get automatic error info");

    
    auto_op.vers = 1;
    if (func != auto_op.func1_default)
        auto_op.is_default = false;
    else
        auto_op.is_default = true;
    auto_op.func1 = func;

    if (H5E__set_auto(estack, &auto_op, client_data) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't set automatic error info");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 
