/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

H5T_str_t
H5Tget_strpad(hid_t type_id)
{
    H5T_t    *dt = NULL;
    H5T_str_t ret_value;

    FUNC_ENTER_API(H5T_STR_ERROR)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_STR_ERROR, "not a datatype");
    while (dt->shared->parent && !H5T_IS_STRING(dt->shared))
        dt = dt->shared->parent; 
    if (!H5T_IS_STRING(dt->shared))
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, H5T_STR_ERROR, "operation not defined for datatype class");

    
    if (H5T_IS_FIXED_STRING(dt->shared))
        ret_value = dt->shared->u.atomic.u.s.pad;
    else
        ret_value = dt->shared->u.vlen.pad;

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Tset_strpad(hid_t type_id, H5T_str_t strpad)
{
    H5T_t *dt        = NULL;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, FAIL, "datatype is read-only");
    if (strpad < H5T_STR_NULLTERM || strpad >= H5T_NSTR)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "illegal string pad type");
    while (dt->shared->parent && !H5T_IS_STRING(dt->shared))
        dt = dt->shared->parent; 
    if (!H5T_IS_STRING(dt->shared))
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "operation not defined for datatype class");

    
    if (H5T_IS_FIXED_STRING(dt->shared))
        dt->shared->u.atomic.u.s.pad = strpad;
    else
        dt->shared->u.vlen.pad = strpad;

done:
    FUNC_LEAVE_API(ret_value)
}
