% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vlsig.R
\name{vlsig}
\alias{vlsig}
\alias{inter_vlsig}
\title{Elementary voice leadings}
\usage{
vlsig(set, index = NULL, display_digits = 2, edo = 12, rounder = 10)

inter_vlsig(
  set,
  goal = NULL,
  index = NULL,
  type = c("ascending", "commontone", "obverse"),
  display_digits = 2,
  edo = 12,
  rounder = 10
)
}
\arguments{
\item{set}{Numeric vector of pitch-classes in the set}

\item{index}{Integer: which voice-leading generator should be displayed? Defaults to \code{NULL},
displaying all voice leadings.}

\item{display_digits}{Integer: how many digits to display when naming any
non-integral interval sizes. Defaults to 2.}

\item{edo}{Number of unit steps in an octave. Defaults to \code{12}.}

\item{rounder}{Numeric (expected integer), defaults to \code{10}:
number of decimal places to round to when testing for equality.}

\item{goal}{For \code{inter_vlsig()} only, vector of the transposition type to voice lead to.
Defaults to \code{NULL}, producing voice leadings to the inversion of \code{set}.}

\item{type}{For \code{inter_vlsig()} only. String: "ascending", "commontone", or "obverse".
Defaults to "ascending", which makes the result prefer ascending voice leadings (as for \code{vlsig()}).
The second makes the result prefer common tones (as might be expected for
contextual inversions). The third option, "obverse", gives the obverse of a voice-leading
in a sense that generalizes Morris (1998, \doi{doi:10.2307/746047})'s concept for
Neo-Riemannian PLR transformations. This option returns voice leadings that lead \emph{to} \code{set} rather
than away from it.}
}
\value{
List with three elements:
\itemize{
\item "vl" which shows the distance (in \code{edo} steps) that each voice moves,
\item "tn" which indicates the (chromatic) transposition achieved by the voice leading,
\item "rotation" which indicates the scalar transposition caused by the voice leading.
}

If \code{index=NULL}, returns instead a matrix whose rows are
all the elementary voice leadings.
}
\description{
Calculate elementary voice leadings which represent motion by a single
arrow on a \code{\link[=brightnessgraph]{brightnessgraph()}}. \code{vlsig()} finds "\strong{v}oice-\strong{l}eading \strong{sig}nature"
of a set moving to transpositions of itself, as determined by \code{\link[=vl_generators]{vl_generators()}}. \code{inter_vlsig()}
finds the elementary voice leadings from a set to some other set, i.e. where the \code{goal}
parameter of \code{\link[=brightnessgraph]{brightnessgraph()}} is not \code{NULL}. By default, \code{inter_vlsig()} finds
voice leadings for contextual inversions of a set.
}
\details{
Note that the voice leadings determined by \code{vlsig()} can be different from the
corresponding ones at the same \eqn{T_n} level in \code{\link[=vl_rolodex]{vl_rolodex()}}. The latter function
prioritizes minimal voice leadings, whereas \code{vlsig()} prioritizes \emph{elementary} voice
leadings derived from a set's \code{\link[=brightnessgraph]{brightnessgraph()}}. In particular, this means that
\code{vlsig()} voice leadings will always be ascending, involve at least one common tone,
and involve no contrary motion. See the \code{odd_pentachord} voice leadings in the Examples.

For \code{vlsig()} the value "rotation" in the result is non-arbitrary: if the rotation value
is n, the voice leading takes \code{set} to the nth mode of \code{set}. For \code{inter_vlsig()}, there
is no canonical correspondence between modes of \code{set} and \code{goal}, except to assume that
the input modes are the 1st mode of each scale. If \code{goal} is \code{NULL}, finding contextual
inversions of \code{set}, the first mode of the inversion is taken to be the one that holds the
first and last pitches of \code{set} in common. These "rotation" values do not have a transparent
relationship to the values of \code{inter_vlsig()}'s index parameter.

For \code{inter_vlsig()} results are not as symmetric between \code{set} and \code{goal} as you might
expect. Since these voice-leading functions study ascending arrows on a brightness graph
the possibilities for \emph{ascending from X to Y} are in principle somewhat different from
the possibilities for \emph{ascending from Y to X}. See the examples for the "Tristan genus."
Note that this is still true when \code{type="commontone"}, which might lead to counterintuitive
results.
}
\examples{
# Hook's elementary signature transformation
major_scale <- c(0, 2, 4, 5, 7, 9, 11)
vlsig(major_scale, index=1)

pure_major_triad <- j(1, 3, 5)
vlsig(pure_major_triad, index=1)
vlsig(pure_major_triad, index=2)

odd_pentachord <- c(0, 1, 4, 9, 11) # in 15-edo
vlsig(odd_pentachord, index=2, edo=15)
vl_rolodex(odd_pentachord, edo=15)$"8" 

# Contextual inversions for Tristan genus:
dom7 <- c(0, 4, 7, 10)
halfdim7 <- c(0, 3, 6, 10)
inter_vlsig(dom7, halfdim7)
inter_vlsig(halfdim7, dom7)

# Elementary voice leadings between unrelated sets:
maj7 <- c(0, 4, 7, 11)
min7 <- c(0, 3, 7, 10)
inter_vlsig(min7, maj7)
brightnessgraph(min7, maj7)

# Elementary inversional VL for just diatonic which is NOT a Q-relation:
inter_vlsig(j(dia), index=3)

# Obverse voice leadings:
# First we see the Parallel transformation which leads from minor to major:
minor <- c(0, 3, 7)
P <- inter_vlsig(minor, index=1)
print(P)
# Compare to its obverse, Slide, leading *to* minor from major:
S <- inter_vlsig(minor, index=1, type="obverse")
print(S)
# A voice-leading plus its obverse is a chromatic transposition:
P$vl + S$vl

}
\seealso{
\code{\link[=vl_generators]{vl_generators()}} and \code{\link[=brightnessgraph]{brightnessgraph()}}
}
