% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ollama_live.R
\name{submit_ollama_pairs_live}
\alias{submit_ollama_pairs_live}
\title{Live Ollama comparisons for a tibble of pairs}
\usage{
submit_ollama_pairs_live(
  pairs,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  host = getOption("pairwiseLLM.ollama_host", "http://127.0.0.1:11434"),
  verbose = TRUE,
  status_every = 1,
  progress = TRUE,
  think = FALSE,
  num_ctx = 8192L,
  include_raw = FALSE,
  ...
)
}
\arguments{
\item{pairs}{Tibble or data frame with at least columns \code{ID1}, \code{text1},
\code{ID2}, \code{text2}. Typically created by \code{\link[=make_pairs]{make_pairs()}}, \code{\link[=sample_pairs]{sample_pairs()}},
and \code{\link[=randomize_pair_order]{randomize_pair_order()}}.}

\item{model}{Ollama model name (for example \code{"mistral-small3.2:24b"},
\code{"qwen3:32b"}, \code{"gemma3:27b"}).}

\item{trait_name}{Trait name to pass to \code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}}.}

\item{trait_description}{Trait description to pass to
\code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}}.}

\item{prompt_template}{Prompt template string, typically from
\code{\link[=set_prompt_template]{set_prompt_template()}}.}

\item{host}{Base URL of the Ollama server. Defaults to the option
\code{getOption("pairwiseLLM.ollama_host", "http://127.0.0.1:11434")}.}

\item{verbose}{Logical; if \code{TRUE}, prints status, timing, and result
summaries.}

\item{status_every}{Integer; print status and timing for every
\code{status_every}-th pair. Defaults to 1 (every pair). Errors are always
printed.}

\item{progress}{Logical; if \code{TRUE}, shows a textual progress bar.}

\item{think}{Logical; see \code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}} for behavior. When
\code{TRUE} and the model name starts with \code{"qwen"}, the temperature is set
to \code{0.6}; otherwise the temperature remains \code{0}.}

\item{num_ctx}{Integer; context window to use via \code{options$num_ctx}. The
default is \code{8192L}.}

\item{include_raw}{Logical; if \code{TRUE}, each row of the returned tibble will
include a \code{raw_response} list-column with the parsed JSON body from
Ollama.}

\item{...}{Reserved for future extensions and forwarded to
\code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}}.}
}
\value{
A tibble with one row per pair and the same columns as
\code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}}, including an optional \code{raw_response}
column when \code{include_raw = TRUE}.
}
\description{
\code{submit_ollama_pairs_live()} is a thin row-wise wrapper around
\code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}}. It takes a tibble of pairs (\code{ID1} / \code{text1} /
\code{ID2} / \code{text2}), submits each pair to a local Ollama server, and binds
the results into a single tibble.
}
\details{
This helper mirrors \code{\link[=submit_openai_pairs_live]{submit_openai_pairs_live()}} but targets a local
Ollama instance rather than a cloud API. It is intended to offer a similar
interface and return shape, so results can be passed directly into
\code{\link[=build_bt_data]{build_bt_data()}} and \code{\link[=fit_bt_model]{fit_bt_model()}}.

Temperature and context length are controlled as follows:
\itemize{
\item By default, \code{temperature = 0} for all models.
\item For Qwen models (model names beginning with \code{"qwen"}) and \code{think = TRUE},
\code{temperature} is set to \code{0.6}.
\item The context window is set via \code{options$num_ctx}, which defaults to
\code{8192} but may be overridden via the \code{num_ctx} argument.
}

In most user-facing workflows, it is more convenient to call
\code{\link[=submit_llm_pairs]{submit_llm_pairs()}} with \code{backend = "ollama"} rather than using
\code{submit_ollama_pairs_live()} directly. The backend-neutral wrapper will
route arguments to the appropriate backend helper and ensure a consistent
return shape.

As with \code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}}, this function assumes that:
\itemize{
\item An Ollama server is running and reachable at \code{host}.
\item The requested models have been pulled in advance (for example
\verb{ollama pull mistral-small3.2:24b}).
}
}
\examples{
\dontrun{
# Requires a running Ollama server and locally available models.

data("example_writing_samples", package = "pairwiseLLM")

pairs <- example_writing_samples |>
  make_pairs() |>
  sample_pairs(n_pairs = 5, seed = 123) |>
  randomize_pair_order(seed = 456)

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# Live comparisons for multiple pairs using a Mistral model via Ollama
res_mistral <- submit_ollama_pairs_live(
  pairs             = pairs,
  model             = "mistral-small3.2:24b",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  verbose           = TRUE,
  status_every      = 2,
  progress          = TRUE
)

res_mistral$better_id

# Qwen with thinking enabled
res_qwen_think <- submit_ollama_pairs_live(
  pairs             = pairs,
  model             = "qwen3:32b",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  think             = TRUE,
  num_ctx           = 16384,
  verbose           = FALSE,
  progress          = FALSE
)

res_qwen_think$better_id
}

}
\seealso{
\itemize{
\item \code{\link[=ollama_compare_pair_live]{ollama_compare_pair_live()}} for single-pair Ollama comparisons.
\item \code{\link[=submit_llm_pairs]{submit_llm_pairs()}} for backend-agnostic comparisons over tibbles of
pairs.
\item \code{\link[=submit_openai_pairs_live]{submit_openai_pairs_live()}}, \code{\link[=submit_anthropic_pairs_live]{submit_anthropic_pairs_live()}},
and \code{\link[=submit_gemini_pairs_live]{submit_gemini_pairs_live()}} for other backend-specific
implementations.
}
}
