% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/graphs.R
\name{rrda.top}
\alias{rrda.top}
\title{Top feature interactions visualization with rank and lambda penalty}
\usage{
rrda.top(
  Y,
  X,
  nrank = NULL,
  lambda = NULL,
  mx = 20,
  my = 20,
  scaling = c("symmetric", "none", "x", "y"),
  title = TRUE
)
}
\arguments{
\item{Y}{A numeric matrix of response variables.}

\item{X}{A numeric matrix of explanatory variables.}

\item{nrank}{Integer rank \eqn{r} of \eqn{\hat{B}} to visualize. If \code{NULL} (default), it is set to \code{min(5, min(dim(X), dim(Y)))}.}

\item{lambda}{A numeric vector of ridge penalty values. If \code{NULL} (default), it is set to 1.}

\item{mx}{Integer; number of top \eqn{X}-features (predictors) to display. Defaults to \code{20}.}

\item{my}{Integer; number of top \eqn{Y}-features (responses) to display. Defaults to \code{20}.}

\item{scaling}{Character string specifying how to apply the singular values from the compositions of \eqn{\hat{B}(\lambda)} when constructing the biplot factors.
Options are:
\code{"symmetric"} (default) distributes singular values evenly to both sides (balanced scaling),
\code{"x"} applies them fully to the X (left) side,
\code{"y"} applies them fully to the Y (right) side,
and \code{"none"} removes them (no singular value weighting).}

\item{title}{Figure title. If \code{TRUE} (default), a formatted title is used. If \code{FALSE} or \code{NULL}, no title is drawn. If a single string, it is passed through to the figure title.}
}
\value{
A list with elements: \code{heatmap} (pheatmap object), \code{B_sub} (mx x my matrix), \code{top_x}, \code{top_y}, \code{b1_sub}, \code{b2_sub}, \code{fit}, \code{scaling}.
}
\description{
Visualizes the most influential feature interactions (based on the L2 norm) from Ridge Redundancy Analysis (RRDA) as a heatmap.

Let the (rank-\eqn{r} truncated) decomposition of \eqn{\hat{B}(\lambda)} be
\deqn{\hat{B}(\lambda, r) = U_{\hat{B}(\lambda)} \, D_{\hat{B}(\lambda)} \, V_{\hat{B}(\lambda)}^{\prime}.}

The following three biplot scalings are defined:

\strong{Symmetric scaling (default)}:
\deqn{\tilde{F} = U_{\hat{B}(\lambda)} \, D_{\hat{B}(\lambda)}^{1/2}, \qquad
      \tilde{G} = V_{\hat{B}(\lambda)} \, D_{\hat{B}(\lambda)}^{1/2}.}

\strong{X scaling}:
\deqn{\tilde{F} = U_{\hat{B}(\lambda)} \, D_{\hat{B}(\lambda)}, \qquad
      \tilde{G} = V_{\hat{B}(\lambda)}.}

\strong{Y scaling}:
\deqn{\tilde{F} = U_{\hat{B}(\lambda)}, \qquad
      \tilde{G} = V_{\hat{B}(\lambda)} \, D_{\hat{B}(\lambda)}.}

In all three cases, \eqn{\hat{B}(\lambda, r) = \tilde{F} \, \tilde{G}^{\prime}.}

Variable importance is scored by the row-wise \eqn{\ell_2}-norms:
\deqn{s_i^{(\tilde{F})} = \| \tilde{F}_{i,\cdot} \|_2, \qquad
      s_j^{(\tilde{G})} = \| \tilde{G}_{j,\cdot} \|_2.}

Selecting the top \eqn{m_x} predictors and \eqn{m_y} responses yields the submatrices of the scaled factor matrices (each with \eqn{r} columns).

The reduced coefficient submatrix is then
\deqn{\hat{B}_{\mathrm{sub}}(\lambda, r) =
      \tilde{F}_{\mathrm{sub}} \, \tilde{G}_{\mathrm{sub}}^{\prime}.}

The matrix \eqn{\hat{B}_{\mathrm{sub}}(\lambda, r)} retains the dominant low-rank structure and is visualized as a heatmap (with \eqn{m_x = m_y = 20} by default).
}
\examples{
set.seed(10)
simdata<-rdasim1(n = 10,p = 50,q = 50,k = 3) # data generation
X <- simdata$X
Y <- simdata$Y
rrda.top(Y=Y,X=X,nrank=5,lambda=1,mx=20,my=20)

\dontrun{
### In practice, the parameters nrank and lambda should be selected by CV ###
cv_result<- rrda.cv(Y = Y, X = X, maxrank = 5, nfold = 5) # cv
best_lambda<-cv_result$opt_min$lambda
best_rank<-cv_result$opt_min$rank
rrda.summary(cv_result = cv_result)

rrda.top(Y=Y,X=X,nrank=best_rank,lambda=best_lambda,mx=20,my=20)
}

}
