% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_pts_and_tox.R
\name{get_pts_and_tox}
\alias{get_pts_and_tox}
\title{Generate Patient and Toxicity Data for BOIN Simulations}
\usage{
get_pts_and_tox(
  n_trials = 10000,
  target,
  p_true,
  n_cohort,
  cohort_size,
  p_saf = NULL,
  p_tox = NULL,
  n_earlystop = 18,
  cutoff_eli = 0.95,
  extrasafe = FALSE,
  offset = 0.05,
  n_earlystop_rule = c("with_stay", "simple"),
  titration = FALSE,
  seed = 123
)
}
\arguments{
\item{n_trials}{Numeric. Number of trials to simulate. Default is 10000.}

\item{target}{Numeric. The target toxicity probability (e.g., 0.30 for 30\%).}

\item{p_true}{Numeric vector. True toxicity probabilities for each dose.}

\item{n_cohort}{Numeric. Maximum number of cohorts per trial.}

\item{cohort_size}{Numeric vector or scalar specifying patients per cohort.}

\item{p_saf}{Numeric. Highest toxicity probability deemed acceptable for safety.
Default is 0.6 * target. Used with p_tox for safety/efficacy dose identification.}

\item{p_tox}{Numeric. Lowest toxicity probability deemed unacceptable for toxicity.
Default is 1.4 * target. Used with p_saf for safety/efficacy dose identification.}

\item{n_earlystop}{Numeric. Sample size at current dose triggering trial termination. Default is 18.}

\item{cutoff_eli}{Numeric. Cutoff probability for dose elimination. Default is 0.95.}

\item{extrasafe}{Logical. If TRUE, apply additional safety stopping rule at the lowest dose. Default is FALSE.}

\item{offset}{Numeric. Offset for safety stopping boundary when extrasafe = TRUE. Default is 0.05.}

\item{n_earlystop_rule}{Character. Rule for early stopping at n_earlystop. Options are "with_stay" or "simple".
Default is "with_stay".}

\item{titration}{Logical. If TRUE, perform accelerated dose escalation in titration phase. Default is FALSE.}

\item{seed}{Numeric. Random seed for reproducibility. Default is 123.}
}
\value{
A list containing:
\item{n_pts_all}{Matrix of size n_trials x n_doses: patient counts at each dose}
\item{n_tox_all}{Matrix of size n_trials x n_doses: DLT counts at each dose}
\item{eliminated_mat}{Matrix of size n_trials x n_doses: logical, TRUE if dose eliminated}
\item{cohorts_completed}{Integer vector of length n_trials: cohorts completed per trial}
\item{stop_reason}{Character vector of length n_trials: reason for stopping}
}
\description{
Conduct vectorized BOIN trial simulations with optional titration phase
to generate patient enrollment and toxicity (DLT) data across multiple trials.
}
\details{
The function simulates multiple BOIN trials in parallel using vectorized operations.
When titration = TRUE, an accelerated dose escalation phase is performed using
single-patient cohorts to rapidly locate the dose-toxicity region. The main trial
then proceeds with standard cohort-based dosing.

Early stopping rules include:
\itemize{
\item Simple rule: Stop when n_earlystop patients are treated at current dose
\item With-stay rule: Stop when n_earlystop patients are treated and dose stays
\item Extra-safety rule (if extrasafe = TRUE): Stop if lowest dose is excessively toxic
}
}
\examples{
\donttest{
# Basic BOIN simulation
target <- 0.30
p_true <- c(0.10, 0.25, 0.40, 0.55, 0.70)

result <- get_pts_and_tox(
  n_trials = 1000,
  target = target,
  p_true = p_true,
  n_cohort = 10,
  cohort_size = 3,
  n_earlystop = 18,
  seed = 123
)

# With titration phase
result_titration <- get_pts_and_tox(
  n_trials = 1000,
  target = target,
  p_true = p_true,
  n_cohort = 10,
  cohort_size = 3,
  titration = TRUE,
  seed = 123
)

# With extra safety stopping
result_safe <- get_pts_and_tox(
  n_trials = 1000,
  target = target,
  p_true = p_true,
  n_cohort = 10,
  cohort_size = 3,
  extrasafe = TRUE,
  offset = 0.05,
  seed = 123
)
}

}
\references{
Liu S. and Yuan, Y. (2015). Bayesian Optimal Interval Designs for Phase I Clinical
Trials. Journal of the Royal Statistical Society: Series C, 64, 507-523.
}
